/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "ReadoutClock.h"
#include "Sensor.h"

namespace sixte {

void ReadoutClock::jumpToNextFrame(ShiftArray& sensor, double tstop) {
  // Start a new frame
  current_frame_number_++;
  current_frame_start_time_ = current_clock_time_;

  // If there has been no photon interaction during last frame,
  // skip frames until frame that includes tstop
  // TODO: Double check which times are really needed

  if (!sensor.anySignal()) {
    long nframes = (long) ((tstop - current_frame_start_time_) / frame_duration_);
    if (nframes == 0) return;

    current_frame_number_ += nframes;
    current_clock_time_ += static_cast<double>(nframes) * frame_duration_; // TODO: calculate as t_0+current_frame_number_*frame_duration_
    current_frame_start_time_ = current_clock_time_;

    for (size_t lineindex = 0; lineindex < sensor.numLines(); lineindex++ ) {
      sensor.addLastReadoutTime(lineindex, static_cast<double>(nframes) * frame_duration_);
    }
  }
}

ReadoutClock::ReadoutClock(XMLData& xml_data) {
  // Initialize from readout node.
  auto readout_node = xml_data.child("detector").child("readout");
  auto children = readout_node.allChildren();

  // Sanity check: between any two <lineshift/> (until <newframe/>),
  // there must be at least one <readoutline lineindex="0" .../>.
  bool need_readout0 = false; // set after a lineshift; cleared by readoutline(0) or newframe
  for (const auto& child : children) {
    const std::string name = child.name();
    if (name == "newframe") {
      need_readout0 = false;
    } else if ((name == "readoutline") || (name == "clearline")) {
      if (child.attributeAsInt("lineindex") == 0) need_readout0 = false;
    } else if (name == "lineshift") {
      if (need_readout0) {
        throw SixteException(
          "Invalid readout sequence: 'lineshift' occurs without an intervening "
          "'readoutline' with lineindex=0 since the previous 'lineshift' (and before the next 'newframe').");
      }
      need_readout0 = true;
    }
  }
  // Wrap-around check.
  if (need_readout0) {
    for (const auto& child : children) {
      const std::string name = child.name();
      if (name == "newframe") { need_readout0 = false; break; }
      if (name == "readoutline" && child.attributeAsInt("lineindex") == 0) { need_readout0 = false; break; }
      if (name == "lineshift") {
        throw SixteException(
          "Invalid readout sequence across boundary: 'lineshift' follows a prior 'lineshift' "
          "without a 'readoutline' (lineindex=0) or 'newframe' in between.");
      }
    }
  }

  // Build the clock list only after validation succeeds.
  for (const auto& child : children) {
    clock_list_.addElement(getNewClockListElement(child));
  }
}

ReadoutClock::ClockListElement ReadoutClock::getNewClockListElement(const XMLNode& xml_node) {
  std::string node_name = xml_node.name();

  if (node_name == "wait") {
    double wait_time = xml_node.attributeAsDouble("time");
    // Accumulate the amount of time required for one read-out frame.
    frame_duration_ += wait_time;
    return new CLWait(wait_time);
  }
  if (node_name == "lineshift")
    return new CLLineShift();
  if (node_name == "newframe")
    return new CLNewFrame();
  if (node_name == "readoutline") {
    auto lineindex = xml_node.attributeAsInt("lineindex");
    auto readoutindex = xml_node.attributeAsInt("readoutindex");
    auto cl_readoutline = new CLReadoutLine(lineindex, readoutindex);
    if (readoutindex > rawymax_)
      rawymax_ = readoutindex;
    if (readoutindex < rawymin_)
      rawymin_ = readoutindex;

    return cl_readoutline;
  }
  if (node_name == "clearline")
    return new CLClearLine(xml_node.attributeAsInt("lineindex"));

  throw SixteException("Failed to initialize readout sequence from XML. Unknown tag: "
                           + node_name);
}

void ReadoutClock::setReadoutStrategy(std::unique_ptr<ShiftArrayReadoutStrategy>&& strategy) {
  readout_strategy_ = std::move(strategy);
}

double ReadoutClock::currentClockTime() const {
  return current_clock_time_;
}

void ReadoutClock::increaseCurrentClockTime(double delta_t) {
  current_clock_time_ += delta_t;
}

void ReadoutClock::readoutLine(ShiftArray& sensor, unsigned int lineindex, unsigned int readoutindex) {

  readout_strategy_->readoutPixels(sensor, lineindex, readoutindex,
                                   current_clock_time_, current_frame_start_time_,
                                   current_frame_number_);
}

void ReadoutClock::operate(ShiftArray& sensor, double tstop) {
  ClockListElement cl_element = clock_list_.currentElement();
  bool continue_operating = true;

  for (;;) {
    continue_operating = cl_element->doOperation(sensor, *this, tstop);

    if (!continue_operating) return;

    cl_element = clock_list_.nextElement();
  }
}

void ReadoutClock::setStartTime(double tstart) {
  current_clock_time_ = tstart;
  current_frame_start_time_ = tstart;
}

bool CLWait::doOperation(ShiftArray& /*sensor*/, ReadoutClock& readout_clock, double tstop) {
  // Only move to the next element if tstop exceeds the waiting period.
  if (tstop <= readout_clock.currentClockTime() + wait_time_) return false;

  // The wait period has finished. Continue with next element.
  readout_clock.increaseCurrentClockTime(wait_time_);

  return true;
}

bool CLLineShift::doOperation(ShiftArray& sensor, ReadoutClock& /*readout_clock*/, double /*tstop*/) {
  sensor.shiftLines();
  return true;
}

bool CLNewFrame::doOperation(ShiftArray& sensor, ReadoutClock& readout_clock, double tstop) {
  readout_clock.jumpToNextFrame(sensor, tstop);

  return true;
}

bool CLReadoutLine::doOperation(ShiftArray& sensor, ReadoutClock& readout_clock, double /*tstop*/) {
  readout_clock.readoutLine(sensor, lineindex_, readoutindex_);
  return true;
}

bool CLClearLine::doOperation(ShiftArray& sensor, ReadoutClock& /*readout_clock*/, double /*tstop*/) {
  headas_chat(5, "clear line %d\n", lineindex_);
  sensor.clearLine(lineindex_);
  return true;
}

}
