/*
This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "ArrayGeometry.h"
#include "GTICollection.h"
#include "Geometry.h"
#include "NewAttitude.h"
#include "NewEventfile.h"
#include "Ebounds.h"
#include "PatternAnalysis.h"
#include "Pha2PiCorrection.h"
#include "PhotonProjection.h"

namespace sixte {

// Forward declaration to break circular dependency with PhotonImaging.h
enum class ImagingType;

class PostprocessingConfiguration {
 public:
  explicit PostprocessingConfiguration(XMLData& xml_data);

  [[nodiscard]] bool needsPatternAnalysis() const {
    return needs_pattern_analysis_;
  }
  [[nodiscard]] bool needsPhotonProjection() const {
    return needs_photon_projection_;
  }
  [[nodiscard]] bool needsFocalPlaneCoordinates() const {
    return needs_focal_plane_coordinates_;
  }
  [[nodiscard]] bool needsPha2PiCorrection() const {
    return needs_pha2pi_correction_;
  }

 private:
  bool needs_pattern_analysis_;
  bool needs_photon_projection_;
  bool needs_focal_plane_coordinates_;
  bool needs_pha2pi_correction_;

  ImagingType imaging_type_;
  std::string split_type_;
};

void addFocalPlaneCoordinatesRectangular(NewEventfile& event_file,
                              const RectangularArray& array_geometry,
                              const Geometry& absorber_geometry,
                              size_t chip_id);

void processPatternAnalysis(std::optional<PatternAnalysis>& pattern_analysis,
                            const Ebounds& ebounds, NewEventfile& raw_event_file,
                            NewEventfile& event_file);

void processPhotonProjection(std::optional<PhotonProjection>& photon_projection,
                             NewAttitude& attitude, NewEventfile& event_file,
                             double tstart, double tstop);

void processPha2PiCorrection(std::optional<Pha2PiCorrection>& pha2pi_correction,
                             NewEventfile& event_file);

void processFocalPlaneCoordinatesRectangular(bool needs_focal_plane_coordinates,
                                             NewEventfile& event_file,
                                             const RectangularArray& array_geometry,
                                             const Geometry& absorber_geometry,
                                             size_t chip_id);

void processGtiExtension(const GTICollection& gti, fitsfile* fits_ptr);

}  // namespace sixte