/*
This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "Postprocessing.h"
#include "PhotonImaging.h"
#include "SixteCCFits.h"
#include "SixteVector.h"
#include "healog.h"

namespace sixte {

PostprocessingConfiguration::PostprocessingConfiguration(XMLData& xml_data) {
  imaging_type_ = readImagingType(xml_data);

  if (auto split = xml_data.child("detector").optionalChild("split")) {
    split_type_ = split->attributeAsString("type");
  } else {
    split_type_ = "none";
  }

  needs_pattern_analysis_ =
      (split_type_ == "exponential" || split_type_ == "GAUSS");
  const bool is_theseus_sxi =
      (xml_data.root().attributeAsString("telescop") == "THESEUS" &&
       xml_data.root().attributeAsString("instrume") == "SXI");
  needs_photon_projection_ = (imaging_type_ == ImagingType::PSF) ||
                             (is_theseus_sxi && imaging_type_ == ImagingType::RAYTRACER);
  needs_focal_plane_coordinates_ = (imaging_type_ == ImagingType::CODEDMASK);
  needs_pha2pi_correction_ = xml_data.child("telescope").optionalChild("pha2pi").has_value();
}

void addFocalPlaneCoordinatesRectangular(NewEventfile& event_file,
                              const RectangularArray& array_geometry,
                              const Geometry& absorber_geometry,
                              size_t chip_id) {
  auto& fitsfile = event_file.fitsfile();

  if (!fitsfile.checkColExists("DETX")) {
    event_file.addCol2CFITSIOEventFile(0, "DETX", "D", "m");
  }
  if (!fitsfile.checkColExists("DETY")) {
    event_file.addCol2CFITSIOEventFile(0, "DETY", "D", "m");
  }
  if (!fitsfile.checkColExists("CHIP_ID")) {
    event_file.addCol2CFITSIOEventFile(0, "CHIP_ID", "J", "");
  }

  size_t num_rows = fitsfile.getNumRows();
  for (size_t row = 1; row <= num_rows; ++row) {
    size_t rawx, rawy;
    fitsfile.readCol("RAWX", row, rawx);
    fitsfile.readCol("RAWY", row, rawy);

    auto [detx, dety] = array_geometry.getRandPosInPixel(array_geometry.xy2PixId(rawx, rawy));

    SixtePoint det_pos(detx, dety, 0.0);
    auto focal_pos = absorber_geometry.transformDetToFocal(det_pos);

    fitsfile.writeCol("DETX", row, focal_pos.x());
    fitsfile.writeCol("DETY", row, focal_pos.y());
    fitsfile.writeCol("CHIP_ID", row, static_cast<int>(chip_id));
  }
}

void processPatternAnalysis(std::optional<PatternAnalysis>& pattern_analysis,
                            const Ebounds& ebounds, NewEventfile& raw_event_file,
                            NewEventfile& event_file) {
  if (pattern_analysis) {
    healog(3) << "start event pattern analysis ...\n";
    pattern_analysis->phpat(ebounds, raw_event_file, event_file);
  } else {
    copyEvents(raw_event_file, event_file);
  }
}

void processPhotonProjection(std::optional<PhotonProjection>& photon_projection,
                             NewAttitude& attitude, NewEventfile& event_file,
                             double tstart, double tstop) {
  if (photon_projection) {
    healog(3) << "start sky projection ...\n";
    photon_projection->doRectangularPhotonProjection(attitude, event_file,
                                                     tstart, tstop);
  }
}

void processPha2PiCorrection(std::optional<Pha2PiCorrection>& pha2pi_correction,
                             NewEventfile& event_file) {
  if (pha2pi_correction) {
    pha2pi_correction->correctEventfile(event_file);
  }
}

void processFocalPlaneCoordinatesRectangular(bool needs_focal_plane_coordinates,
                                             NewEventfile& event_file,
                                             const RectangularArray& array_geometry,
                                             const Geometry& absorber_geometry,
                                             size_t chip_id) {
  if (needs_focal_plane_coordinates) {
    healog(3) << "add focal plane coordinates ...\n";
    addFocalPlaneCoordinatesRectangular(event_file, array_geometry,
                             absorber_geometry, chip_id);
  }
}

void processGtiExtension(const GTICollection& gti, fitsfile* fits_ptr) {
  gti.saveGtiExtension(fits_ptr);
}

}  // namespace sixte
