/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "PhotonProjection.h"

#include "sixte_random.h"

namespace sixte {

PhotonProjection::PhotonProjection(XMLData& xml_data) {
  auto detector = xml_data.child("detector");
  // Check if geometry node exists, otherwise use detector directly
  // (for backward compatibility)
  auto parent_node = detector.optionalChild("geometry").value_or(detector);
  
  auto wcs = parent_node.child("wcs");
  double rota = wcs.attributeAsDoubleOr("rota", 0.0);
  rotation_angle_ = rota * M_PI / 180.;

  xrpix_ = wcs.attributeAsDouble("xrpix");
  yrpix_ = wcs.attributeAsDouble("yrpix");

  xrval_ = wcs.attributeAsDouble("xrval");
  yrval_ = wcs.attributeAsDouble("yrval");

  xdelt_ = wcs.attributeAsDouble("xdelt");
  ydelt_ = wcs.attributeAsDouble("ydelt");

  focal_length_ = xml_data.child("telescope").child("focallength").attributeAsDouble("value");
  
  cosrota_ = cos(rotation_angle_);
  sinrota_ = sin(rotation_angle_);

  if (xml_data.root().attributeAsString("telescop") == "THESEUS"
      && xml_data.root().attributeAsString("instrume") == "SXI" ) {
      isTheseus_ = true;
  }
}

//void PhotonProjection::rectPhproj(NewAttitude& attitude,
//                                  NewEventfile& eventfile,
//                                  double tstart, double tstop) const {
//
//  size_t nrows = eventfile.getRowNum(1U);
//
//  auto buffering = getEventsBuffer(nrows);
//  size_t buffer_loop = buffering.first;
//  size_t buffer_size = buffering.second;
//
//  size_t row = 1;
//  for (size_t buffer_index=0; buffer_index < buffer_loop; buffer_index++) {
//    std::vector<NewEvent> events;
//
//    if (buffer_index+1 != buffer_loop) events = eventfile.getEventsFromFile(row, buffer_size, 1U);
//    else events = eventfile.getEventsFromFile(row, nrows, 1U);
//
//    for (auto &event : events) {
//      if (event.time < tstart) continue;
//      if (event.time > tstop) break;
//
//      Telescope_attitude telescope_attitude = attitude.getTelescopeAxes(event.time);
//
//      auto ra_dec = projectRectEvent(event.rawx, event.rawy, telescope_attitude);
//
//      event.ra = ra_dec.first;
//      event.dec = ra_dec.second;
//    }
//    eventfile.updateEventsInFile(1, events, 1U);
//    nrows -= buffer_size;
//    row += buffer_size;
//  }
//}

void PhotonProjection::doRectangularPhotonProjection(NewAttitude& attitude,
                                                     NewEventfile& eventfile,
                                                     double tstart,
                                                     double tstop) const {
  
  size_t numrows = eventfile.getRowNumCFITSIO("EVENTS");
  
  for (size_t row = 1; row <= numrows; row++) {
    NewEvent event = eventfile.getEventFromCFITSIOFile(row);

    // Check whether we are still within the requested time interval.
    if (event.time < tstart) continue;
    if (event.time > tstop) break;
    
    Telescope_attitude telescope_attitude = attitude.getTelescopeAxes(event.time);
    
    if (isTheseus_) {
      std::vector<double> origin_vec;
      eventfile.fitsfile().readCol("ORIGINVEC", row, 3, origin_vec);

      SixteVector origin(origin_vec[0], origin_vec[1], origin_vec[2]);

      auto [ra, dec] = projectRectTheseusEvent(origin, telescope_attitude);
      event.ra = ra;
      event.dec = dec;
    } else {
    auto [ra, dec] = projectRectEvent(event.rawx, event.rawy, telescope_attitude);
    event.ra = ra;
    event.dec = dec;
    }

    eventfile.updateEvent2CFITSIOFile(event, row);
  }
}

std::pair<double, double> PhotonProjection::projectRectEvent(size_t rawx, size_t rawy,
                                                         const Telescope_attitude& telescope_attitude) const {
  
  auto [detx, dety] = getRectRandPosInPixel(rawx, rawy);
  
  double x_rota = cosrota_ * detx - sinrota_ * dety;
  double y_rota = sinrota_ * detx + cosrota_ * dety;

  double detpos_x = x_rota + xrval_; // in [m]
  double detpos_y = y_rota + yrval_; // in [m]
  
  SixteVector srcpos = telescope_attitude.nz
    + (detpos_x / focal_length_ * telescope_attitude.nx)
    + (detpos_y / focal_length_ * telescope_attitude.ny);
  srcpos = normalizeVector(srcpos);
  
  return calculateRaDec(srcpos);
}

std::pair<double, double> PhotonProjection::projectRectTheseusEvent(const SixteVector& origin,
                                                                    const Telescope_attitude& telescope_attitude) {
  auto origin_norm = normalizeVector(origin); // TODO: Check if needed

  SixteVector srcpos = origin_norm.x() * -1 * telescope_attitude.nx
                     + origin_norm.y() * -1 * telescope_attitude.ny
                     + origin_norm.z() * telescope_attitude.nz;

  auto norm_srcpos = normalizeVector(srcpos);

  return calculateRaDec(norm_srcpos);
}

std::pair<double,double> PhotonProjection::getRectRandPosInPixel(size_t rawx, size_t rawy) const {
  
  double x_off = getUniformRandomNumber();
  double y_off = getUniformRandomNumber();
  
  double x_rand_pos_in_pix = ((double)rawx - xrpix_ + 0.5 + x_off) * xdelt_;
  double y_rand_pos_in_pix = ((double)rawy - yrpix_ + 0.5 + y_off) * ydelt_;
  
  return std::make_pair(x_rand_pos_in_pix, y_rand_pos_in_pix);
}

void doMicroCalPhotonProjection(NewAttitude& attitude, const Geometry& absorber_geometry,
                                const ArrayGeometry& array_geometry,
                                double focal_length, fitsfile *fptr, double tstart, double tstop) {
  // TODO: Use Fitsfile class instead of fptr

  cfitsio::moveToExt(fptr, "EVENTS");
  size_t numrows = cfitsio::getNumRows(fptr);
  
  unsigned long pix_id;
  double time;

  auto colnum_pixid = cfitsio::getColNum(fptr, "PIXID");
  auto colnum_time = cfitsio::getColNum(fptr, "TIME");
  auto colnum_ra = cfitsio::getColNum(fptr, "RA");
  auto colnum_dec = cfitsio::getColNum(fptr, "DEC");
  auto colnum_detx = cfitsio::getColNum(fptr, "DETX");
  auto colnum_dety = cfitsio::getColNum(fptr, "DETY");

  for (size_t row = 1; row <= numrows; row++) {
    cfitsio::fitsReadCol(fptr, colnum_pixid, row, 1, &pix_id);
    cfitsio::fitsReadCol(fptr, colnum_time, row, 1, &time);
    
    // Check whether we are still within the requested time interval.
    if (time < tstart) continue;
    if (time > tstop) break;
    
    // determine position within pixel
    auto [detx, dety] = array_geometry.getRandPosInPixel(pix_id-1); // PIXIDs are 1-based in FITS file!
    
    // get sky coordinates and write
    auto [ra, dec] = detToSky(detx, dety, focal_length, time, attitude, absorber_geometry);
    ra *= 180/M_PI;
    dec *= 180/M_PI;

    cfitsio::fitsWriteCol(fptr, colnum_ra, row, 1, &ra);
    cfitsio::fitsWriteCol(fptr, colnum_dec, row, 1, &dec);
    cfitsio::fitsWriteCol(fptr, colnum_detx, row, 1, &detx);
    cfitsio::fitsWriteCol(fptr, colnum_dety, row, 1, &dety);
  }
}

} // sixte
