/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <boost/random.hpp>
#include <memory>
#include <string>

#include "Carrier.h"
#include "Ebounds.h"
#include "EnergyDeposition.h"
#include "NewRMF.h"
#include "SixteException.h"
#include "SixtePhoton.h"
#include "StochInterp.h"
#include "XMLData.h"

namespace sixte {

class PhotonInteractionStrategy {
 public:
  virtual ~PhotonInteractionStrategy() = default;
  [[nodiscard]] virtual EnergyDepositions doInteraction(const Geometry& absorber_geometry,
                                                        const SixtePhoton& photon) const = 0;
  [[nodiscard]] virtual std::shared_ptr<const Ebounds> canonicalEbounds() const { return nullptr; }
};

class RMFBasedInteraction: public PhotonInteractionStrategy {
 public:
  /**
   * Initializes an interaction strategy that uses an RMF based approach to
   * create a gaussian charge cloud from a photon impact.
   * The charge cloud sigma is calculated as
   * sigma(E_ph) = cloud_par1 + cloud_par2*sqrt(E_ph)
   *
   * @param xml_data    An initialized XMLData.
   */
  explicit RMFBasedInteraction(std::shared_ptr<const NewRMF> rmf);


  /**
   * Implements doInteraction function of this interaction strategy.
   *
   * @param photon    A photon impact on the absorber.
   * @return          The charge cloud created from interaction of the photon
   *                  with absorber.
   */
  [[nodiscard]] EnergyDepositions doInteraction(const Geometry& absorber_geometry,
                                                const SixtePhoton &photon) const override;

  [[nodiscard]] std::shared_ptr<const Ebounds> canonicalEbounds() const override { return canonical_ebounds_; }

 private:
  std::shared_ptr<const NewRMF> rmf_;
  std::shared_ptr<const Ebounds> canonical_ebounds_;
};

class FullAbsorption: public PhotonInteractionStrategy {

  /**
   * Implements doInteraction function of this interaction strategy.
   *
   * @param photon    A photon impact on the absorber.
   * @return          The charge cloud created from interaction of the photon
   *                  with absorber.
   */
  [[nodiscard]] EnergyDepositions doInteraction(
      const Geometry& absorber_geometry,
      const SixtePhoton &photon) const override;

};

class AngleDependentRMFBasedInteraction final: public PhotonInteractionStrategy {
 public:
  AngleDependentRMFBasedInteraction(const XMLData& xml_data,
                                    const std::shared_ptr<RmfRegistry>& registry);

  [[nodiscard]] EnergyDepositions doInteraction(const Geometry& absorber_geometry,
                                                const SixtePhoton& photon) const override;

  [[nodiscard]] std::shared_ptr<const Ebounds> canonicalEbounds() const override {
    return canonical_ebounds_;
  }

 private:
  static double foldPhiFourfold(double phi_deg);
  static void validateEboundsUniform(const std::vector<std::shared_ptr<const NewRMF>>& nodes);

  StochInterp<2, std::shared_ptr<const NewRMF>> interp_{{0, 0}};
  std::shared_ptr<const Ebounds> canonical_ebounds_;
};

}
