/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <string>

#include "BoundingBox.h"
#include "CodedMask.h"
#include "NewAttitude.h"
#include "NewImpactfile.h"
#include "NewPSF.h"
#include "ObsInfo.h"
#include "Parameters.h"
#include "SixteException.h"
#include "SixtePhoton.h"
#include "XMLData.h"

#include "raytracing/mirror_module/MirrorModule.h"
#include "raytracing/mirror_module/Wolter.h"
#include "raytracing/mirror_module/LobsterEyeOptic.h"

namespace sixte {

enum class RayRerollPolicy {
  NONE,
  ALL_FAILURES,
  OPTICS_ONLY,
  IMPACT_POSITION_ONLY,
};

class PSFImagingStrategy {
 public:
  explicit PSFImagingStrategy(XMLData& xml_data);

  std::optional<SixtePhoton> operator()(NewAttitude& attitude,
                                        const SixtePhoton& photon);

 private:
  bool write_impact_file_ = false;
  double fov_diameter_;
  double focal_length_;
  NewPSF psf_;
  Telescope_attitude telescope_{};
};

class CodedMaskImagingStrategy {
 public:
  /**
   * Create coded mask imaging strategy from XML.
   * @param xml_data XML containing telescope and mask parameters
   */
  explicit CodedMaskImagingStrategy(XMLData& xml_data);

  std::optional<SixtePhoton> operator()(NewAttitude& attitude,
                                        const SixtePhoton& photon);

 private:
  /**
   * Calculate photon impact position using coded mask imaging.
   * @param theta Off-axis angle from optical axis [rad]
   * @param phi Azimuthal angle around optical axis [rad]
   * @return Impact position in focal plane coordinates [m]
   */
  [[nodiscard]] std::optional<SixtePoint> calculateImpactPosition(double theta, double phi) const;

  [[nodiscard]] static std::vector<Rectangle2d> parseDetectorRectsFocal(const XMLData& xml_data) ;
  [[nodiscard]] std::optional<SixtePoint> sampleImpactPositionFromOverlap(
    const Rectangle2d& illuminated_rect, double shift_x, double shift_y) const;

  [[nodiscard]] std::optional<SixtePoint> sampleRandomDetectorPosition() const;

  [[nodiscard]] double imagingProbability(double energy_keV) const noexcept;
  [[nodiscard]] bool shouldImagePhoton(double energy_keV) const;
  void buildDetectorAreaCdf();

  CodedMask coded_mask_;
  double focal_length_{0.0};   ///< Distance from mask to detector [m]
  double fov_diameter_{0.0};   ///< Field of view diameter [rad]

  std::optional<NewVignetting> vignetting_;
  std::vector<Rectangle2d> detector_rects_focal_;
  std::vector<double> detector_area_cdf_;
  double detector_total_area_{0.0};

  // Caches to avoid per-photon allocations in the reroll path.
  mutable std::vector<Rectangle2d> overlap_rects_cache_;
  mutable std::vector<double> overlap_area_cdf_cache_;

  size_t max_reroll_{10000};
  RayRerollPolicy reroll_policy_{RayRerollPolicy::NONE};

  bool high_energy_transparency_enabled_{false};
  double high_energy_transparency_e_low_keV_{0.0};
  double high_energy_transparency_e_high_keV_{0.0};
  bool high_energy_transparency_apply_fov_{true};
  bool high_energy_transparency_apply_vignetting_{false};
};

class RaytracingImagingStrategy {
 public:
  explicit RaytracingImagingStrategy(XMLData& xml_data);

    // 1) Copy‐ctor: deep‐clone the mirror_module_
    RaytracingImagingStrategy(const RaytracingImagingStrategy& o)
      : mirror_module_(o.mirror_module_->clone()),
        reroll_policy_(o.reroll_policy_),
        fov_(o.fov_)
    {}

    // 2) Copy‐assignment
    RaytracingImagingStrategy& operator=(const RaytracingImagingStrategy& o) {
      if (this != &o) {
        mirror_module_  = o.mirror_module_->clone();
        reroll_policy_  = o.reroll_policy_;
        fov_            = o.fov_;
      }
        return *this;
    }

    // 3) Move‐ctor
    RaytracingImagingStrategy(RaytracingImagingStrategy&&) noexcept = default;

    // 4) Move‐assignment
    RaytracingImagingStrategy& operator=(RaytracingImagingStrategy&&) noexcept = default;

    // 5) Destructor
    ~RaytracingImagingStrategy() = default;

  std::optional<SixtePhoton> operator()(NewAttitude& attitude,
                                        const SixtePhoton& photon);
 private:
  static std::unique_ptr<MirrorModule> create_telescope(XMLData& xml_data);
  std::unique_ptr<MirrorModule> mirror_module_;

  struct RayTraceResult {
    std::optional<Ray> ray;
    RayFailReason fail_reason = RayFailReason::UNSET;
  };

  [[nodiscard]] bool shouldReroll(RayFailReason reason) const;

  RayTraceResult LaunchRay(Vec3fa &photon_direction,
                           const sixte::SixtePhoton &photon);

  RayTraceResult RayTraceOpticsOnce(Vec3fa &photon_direction,
                                    const sixte::SixtePhoton &photon);

  RayTraceResult RayTraceFullOnce(Vec3fa &photon_direction,
                                  const sixte::SixtePhoton &photon);

  /**
   * To reproduce an ARF containing all relevant effects, we have to reroll
   * every photon until we get a valid result,
   * or we rolled too often (safety feature)
   * @param photon_direction Initial photon direction
   * @param photon Sixte Photon
   * @return optional Ray
   */
  std::optional<Ray> RayTraceReroll(Vec3fa& photon_direction,
                                    const sixte::SixtePhoton &photon);

  /**
   * Get randomized position in the telescopes field of view
   * + 10% margin. Needs to be converted from m to mm.
   * @return randomized pair of double
   */
  [[nodiscard]] std::pair<double, double> GetRandomPhotonPosInFOV() const;

  RayRerollPolicy reroll_policy_;
  double fov_;
};

enum class ImagingType { PSF, CODEDMASK, RAYTRACER };

ImagingType readImagingType(XMLData& xml_data);
size_t readMaxReroll(XMLData& xml_data);

using ImagingStrategy =
    std::function<std::optional<SixtePhoton>(NewAttitude&, const SixtePhoton&)>;

class PhotonImaging {
 public:
  PhotonImaging(const std::string& output_file, bool clobber, XMLData& xml_data,
                const ObsInfo& obs_info);

  std::optional<SixtePhoton> doImaging(NewAttitude& attitude,
                                       const SixtePhoton& photon);
  void checkIfImaged() const;

 private:
  static ImagingStrategy createImagingStrategy(XMLData& xml_data);

  ImagingStrategy imaging_strategy_;
  bool write_impact_file_ = false;
  size_t num_imaged_{0};
  std::optional<ImpactFile> impact_file_;
};

bool isPhotonInFov(const SixtePhoton& photon, NewAttitude& attitude, double fov_diameter);
}  // namespace sixte
