/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <algorithm>
#include <cctype>
#include <cmath>
#include <numeric>
#include <string>

#include "PhotonImaging.h"
#include "healog.h"
#include "sixte_random.h"

namespace sixte {

namespace {

std::pair<double, double> computeIncidentAnglesRad(const SixtePhoton& photon,
                                                   const Telescope_attitude& telescope) {
  auto source_direction = unitVector(*photon.ra(), *photon.dec());
  double theta = calculateOffAxisAngleTheta(telescope.nz, source_direction);
  double phi = calculateAzimuthalAnglePhi(telescope.nx, telescope.ny, source_direction);

  // Normalize phi to [0, 2*PI).
  while (phi < 0) phi += 2 * M_PI;
  while (phi >= 2 * M_PI) phi -= 2 * M_PI;

  return {theta, phi};
}

RayRerollPolicy parseRerollPolicy(const std::string& policy) {
  if (policy == "none") return RayRerollPolicy::NONE;
  if (policy == "all") return RayRerollPolicy::ALL_FAILURES;
  if (policy == "optics_only") return RayRerollPolicy::OPTICS_ONLY;
  if (policy == "impact_position_only") return RayRerollPolicy::IMPACT_POSITION_ONLY;

  throw SixteException("Unknown reroll_policy='" + policy +
                       "'. Use one of: none, all, optics_only, impact_position_only.");
}

struct RerollPolicySetting {
  RayRerollPolicy policy{RayRerollPolicy::NONE};
  std::optional<std::string> policy_text;
};

RerollPolicySetting readRerollPolicySetting(XMLData& xml_data) {
  RerollPolicySetting setting{};
  auto telescope = xml_data.child("telescope");
  if (!telescope.hasChild("arf")) return setting;

  auto arf = telescope.child("arf");
  if (!arf.hasAttribute("reroll_policy")) {
    healog(5) << "Reroll policy attribute not set in XML, defaulting to none" << std::endl;
    return setting;
  }

  setting.policy_text = arf.attributeAsString("reroll_policy");
  setting.policy = parseRerollPolicy(*setting.policy_text);
  return setting;
}

void validateRerollPolicyForImaging(
    ImagingType imaging_type,
    const RerollPolicySetting& reroll_policy) {
  if (!reroll_policy.policy_text) return;

  const auto& policy_text = *reroll_policy.policy_text;
  if (imaging_type == ImagingType::PSF &&
      reroll_policy.policy != RayRerollPolicy::NONE) {
    throw SixteException("reroll_policy='" + policy_text +
                         "' is not supported for imaging='psf'");
  }
  if (imaging_type == ImagingType::CODEDMASK &&
      reroll_policy.policy == RayRerollPolicy::OPTICS_ONLY) {
    throw SixteException("reroll_policy='" + policy_text +
                         "' is not supported for imaging='codedmask'");
  }
  if (imaging_type == ImagingType::RAYTRACER &&
      reroll_policy.policy == RayRerollPolicy::IMPACT_POSITION_ONLY) {
    throw SixteException("reroll_policy='" + policy_text +
                         "' is not supported for imaging='raytracer'");
  }
}

RayRerollPolicy readValidatedRerollPolicy(XMLData& xml_data, ImagingType imaging_type) {
  const auto reroll_policy = readRerollPolicySetting(xml_data);
  validateRerollPolicyForImaging(imaging_type, reroll_policy);
  return reroll_policy.policy;
}

}  // namespace

PSFImagingStrategy::PSFImagingStrategy(XMLData& xml_data)
    : fov_diameter_(
          xml_data.child("telescope").child("fov").attributeAsDouble("diameter")),
      focal_length_(
          xml_data.child("telescope").child("focallength").attributeAsDouble("value")),
      psf_(xml_data.dirname() +
               xml_data.child("telescope").child("psf").attributeAsString("filename"),
           focal_length_,
           xml_data.dirname() + xml_data.child("telescope").child("vignetting").attributeAsString("filename")) {
  (void)readValidatedRerollPolicy(xml_data, ImagingType::PSF);
  fov_diameter_ *= M_PI / 180;
}

std::optional<SixtePhoton> PSFImagingStrategy::operator()(
    NewAttitude& attitude, const SixtePhoton& photon) {
  if (!isPhotonInFov(photon, attitude, fov_diameter_)) {
    return std::nullopt;
  }

  telescope_ = attitude.getTelescopeAxes(photon.time());
  auto position = psf_.get_NewPSF_pos(photon, telescope_, focal_length_);

  if (!position) {
    return std::nullopt;
  }

  // Check whether the photon hits the detector within the FOV.
  // ToDo: put this in docu:
  // (Due to the effects of the mirrors it might have been scattered over
  // the edge of the FOV, although the source is inside the FOV.)
  double pos_x = position->x();
  double pos_y = position->y();
  if (sqrt(pow(pos_x, 2.) + pow(pos_y, 2.)) >= tan(fov_diameter_) * focal_length_) {
    return std::nullopt;
  }

  SixtePhoton result(photon.time(),
                     photon.energy(),
                     {pos_x, pos_y, 0.},
                     photon.photon_metainfo_ref());

  auto [theta, phi] = computeIncidentAnglesRad(photon, telescope_);
  result.photon_metainfo_ref().setIncidentAnglesRad(theta, phi);

  return result;
}

CodedMaskImagingStrategy::CodedMaskImagingStrategy(XMLData& xml_data)
    : coded_mask_{xml_data},
      focal_length_{xml_data.child("telescope").child("focallength").attributeAsDouble("value")},
      fov_diameter_{xml_data.child("telescope").child("fov").attributeAsDouble("diameter") * M_PI / 180},
      max_reroll_{readMaxReroll(xml_data)},
      reroll_policy_{readValidatedRerollPolicy(xml_data, ImagingType::CODEDMASK)} {
  auto telescope = xml_data.child("telescope");

  if (telescope.hasChild("vignetting")) {
    const auto vig_file = telescope.child("vignetting").attributeAsString("filename");
    vignetting_.emplace(xml_data.dirname() + vig_file);
  }

  if (auto het_node = telescope.optionalChild("high_energy_transparency")) {
    high_energy_transparency_enabled_ = true;
    high_energy_transparency_e_low_keV_ = het_node->attributeAsDouble("e_low");
    high_energy_transparency_e_high_keV_ = het_node->attributeAsDouble("e_high");

    if (high_energy_transparency_e_high_keV_ <= high_energy_transparency_e_low_keV_) {
      throw SixteException(
          "high_energy_transparency requires e_high > e_low, got e_low=" +
          std::to_string(high_energy_transparency_e_low_keV_) +
          " keV, e_high=" +
          std::to_string(high_energy_transparency_e_high_keV_) + " keV");
    }

    high_energy_transparency_apply_fov_ = het_node->attributeAsBoolOr("apply_fov", true);
    high_energy_transparency_apply_vignetting_ = het_node->attributeAsBoolOr("apply_vignetting", true);

    healog(5) << "CodedMask high_energy_transparency enabled: e_low="
              << high_energy_transparency_e_low_keV_ << " keV, e_high="
              << high_energy_transparency_e_high_keV_ << " keV, apply_fov="
              << high_energy_transparency_apply_fov_ << ", apply_vignetting="
              << high_energy_transparency_apply_vignetting_ << std::endl;
  }

  const bool need_detector_rects =
      high_energy_transparency_enabled_ ||
      reroll_policy_ == RayRerollPolicy::ALL_FAILURES ||
      reroll_policy_ == RayRerollPolicy::IMPACT_POSITION_ONLY;

  if (need_detector_rects) {
    detector_rects_focal_ = parseDetectorRectsFocal(xml_data);
    if (detector_rects_focal_.empty()) {
      throw SixteException(
          "Coded-mask imaging requires at least one <detector> geometry with type='rectarray' "
          "when using reroll_policy or high_energy_transparency");
    }

    overlap_rects_cache_.reserve(detector_rects_focal_.size());
    overlap_area_cdf_cache_.reserve(detector_rects_focal_.size());

    if (high_energy_transparency_enabled_) {
      buildDetectorAreaCdf();
    }
  }
}

std::optional<SixtePhoton> CodedMaskImagingStrategy::operator()(
    NewAttitude& attitude, const SixtePhoton& photon) {
  const bool use_coded_mask = shouldImagePhoton(photon.energy());

  const bool require_fov = use_coded_mask || high_energy_transparency_apply_fov_;
  if (require_fov && !isPhotonInFov(photon, attitude, fov_diameter_)) {
    return std::nullopt;
  }

  const auto telescope = attitude.getTelescopeAxes(photon.time());
  const auto [theta, phi] = computeIncidentAnglesRad(photon, telescope);

  const bool apply_vignetting = use_coded_mask || high_energy_transparency_apply_vignetting_;
  if (vignetting_ && apply_vignetting) {
    const auto vig = vignetting_->getVignettingFactor(photon.energy(), theta, phi);
    if (getUniformRandomNumber() > vig) {
      return std::nullopt;
    }
  }

  std::optional<SixtePoint> impact_position;
  if (use_coded_mask) {
    impact_position = calculateImpactPosition(theta, phi);
  } else {
    impact_position = sampleRandomDetectorPosition();
  }

  if (!impact_position) {
    return std::nullopt;
  }

  SixtePhoton result(photon.time(),
                     photon.energy(),
                     *impact_position,
                     photon.photon_metainfo_ref());

  result.photon_metainfo_ref().setIncidentAnglesRad(theta, phi);

  return result;
}

double CodedMaskImagingStrategy::imagingProbability(double energy_keV) const noexcept {
  if (!high_energy_transparency_enabled_) {
    return 1.0;
  }

  if (energy_keV <= high_energy_transparency_e_low_keV_) {
    return 1.0;
  }
  if (energy_keV >= high_energy_transparency_e_high_keV_) {
    return 0.0;
  }

  const auto t = (energy_keV - high_energy_transparency_e_low_keV_) /
                 (high_energy_transparency_e_high_keV_ - high_energy_transparency_e_low_keV_);
  return std::clamp(1.0 - t, 0.0, 1.0);
}

bool CodedMaskImagingStrategy::shouldImagePhoton(double energy_keV) const {
  const auto p = imagingProbability(energy_keV);
  if (p <= 0.0) return false;
  if (p >= 1.0) return true;

  return getUniformRandomNumber() < p;
}

void CodedMaskImagingStrategy::buildDetectorAreaCdf() {
  detector_area_cdf_.clear();
  detector_area_cdf_.reserve(detector_rects_focal_.size());

  detector_total_area_ = 0.0;
  for (const auto& rect : detector_rects_focal_) {
    const auto area = rect.area();
    if (area <= 0.0) {
      throw SixteException(
          "Detector geometry contains a rectangle with non-positive area");
    }
    detector_total_area_ += area;
    detector_area_cdf_.push_back(detector_total_area_);
  }

  if (detector_total_area_ <= 0.0 || detector_area_cdf_.empty()) {
    throw SixteException(
        "High-energy transparency requires a positive total detector area");
  }
}

std::optional<SixtePoint> CodedMaskImagingStrategy::sampleRandomDetectorPosition() const {
  if (detector_rects_focal_.empty() || detector_total_area_ <= 0.0 ||
      detector_area_cdf_.empty()) {
    throw SixteException(
        "High-energy transparency requires detector geometry and precomputed detector area CDF");
  }

  const auto r = getUniformRandomNumber() * detector_total_area_;
  const auto it = std::lower_bound(detector_area_cdf_.begin(), detector_area_cdf_.end(), r);
  const auto idx = static_cast<size_t>(std::distance(detector_area_cdf_.begin(), it));

  const auto& rect = detector_rects_focal_.at(idx);
  const auto x = rect.xmin() + getUniformRandomNumber() * (rect.xmax() - rect.xmin());
  const auto y = rect.ymin() + getUniformRandomNumber() * (rect.ymax() - rect.ymin());

  return SixtePoint{x, y, 0.0};
}

std::optional<SixtePoint> CodedMaskImagingStrategy::calculateImpactPosition(double theta, double phi) const {
  const auto shift_x = focal_length_ * std::tan(theta) * std::cos(phi);
  const auto shift_y = focal_length_ * std::tan(theta) * std::sin(phi);

  if (reroll_policy_ == RayRerollPolicy::ALL_FAILURES ||
      reroll_policy_ == RayRerollPolicy::IMPACT_POSITION_ONLY) {
    // Sample from the intersection between the detector plane and the
    // projected mask area ("illuminated" part), then reroll only the mask sampling
    // until an open pixel is hit.
    const double min_x = -coded_mask_.xWidth() / 2.0 + coded_mask_.xOffset() + shift_x;
    const double max_x = +coded_mask_.xWidth() / 2.0 + coded_mask_.xOffset() + shift_x;
    const double min_y = -coded_mask_.yWidth() / 2.0 + coded_mask_.yOffset() + shift_y;
    const double max_y = +coded_mask_.yWidth() / 2.0 + coded_mask_.yOffset() + shift_y;
    const Rectangle2d illuminated_rect(Point_2(min_x, min_y), Point_2(max_x, max_y));

    return sampleImpactPositionFromOverlap(illuminated_rect, shift_x, shift_y);
  }

  // Default: direct raytracing from an open mask pixel
  auto [mask_pixel_x, mask_pixel_y] = coded_mask_.sampleOpenPixel();
  auto [mask_x, mask_y] = coded_mask_.pixelToPhysical(
      mask_pixel_x, mask_pixel_y, getUniformRandomNumber(), getUniformRandomNumber());

  return SixtePoint{mask_x + shift_x, mask_y + shift_y, 0.0};
}

std::vector<Rectangle2d> CodedMaskImagingStrategy::parseDetectorRectsFocal(
    const XMLData& xml_data) {
  // TODO: Use RectangularArray

  std::vector<Rectangle2d> rects;

  for (const auto& detector : xml_data.children("detector")) {
    const auto parent_node = detector.optionalChild("geometry").value_or(detector);

    const auto geo_type = parent_node.attributeAsStringOr("type", "");
    if (geo_type != "rectarray") {
      throw SixteException(
          "CodedMask reroll currently supports only geometry type='rectarray', got '" +
          geo_type + "'");
    }

    const auto dimensions = parent_node.child("dimensions");
    const auto xwidth = dimensions.attributeAsInt("xwidth");
    const auto ywidth = dimensions.attributeAsInt("ywidth");

    const auto wcs = parent_node.child("wcs");
    const auto xrpix = wcs.attributeAsDouble("xrpix");
    const auto yrpix = wcs.attributeAsDouble("yrpix");
    const auto xrval = wcs.attributeAsDoubleOr("xrval", 0.0);
    const auto yrval = wcs.attributeAsDoubleOr("yrval", 0.0);
    const auto xdelt = wcs.attributeAsDouble("xdelt");
    const auto ydelt = wcs.attributeAsDouble("ydelt");
    const auto rota_deg = wcs.attributeAsDoubleOr("rota", 0.0);
    if (std::abs(rota_deg) > 1e-12) {
      throw SixteException(
          "CodedMask reroll: non-zero WCS rota is currently not supported (rota=" +
          std::to_string(rota_deg) + " deg)");
    }

    const auto xshift = xrpix - 0.5;
    const auto yshift = yrpix - 0.5;

    const auto x1 = (0.0 - xshift) * xdelt + xrval;
    const auto x2 = (static_cast<double>(xwidth) - xshift) * xdelt + xrval;
    const auto y1 = (0.0 - yshift) * ydelt + yrval;
    const auto y2 = (static_cast<double>(ywidth) - yshift) * ydelt + yrval;

    const auto min_x = std::min(x1, x2);
    const auto max_x = std::max(x1, x2);
    const auto min_y = std::min(y1, y2);
    const auto max_y = std::max(y1, y2);

    rects.emplace_back(Point_2(min_x, min_y), Point_2(max_x, max_y));
  }

  return rects;
}

std::optional<SixtePoint> CodedMaskImagingStrategy::sampleImpactPositionFromOverlap(
    const Rectangle2d& illuminated_rect, double shift_x, double shift_y) const {
  // Build overlap rectangles (detector chip rectangles intersected with illuminated area).
  overlap_rects_cache_.clear();
  overlap_area_cdf_cache_.clear();

  double total_area = 0.0;
  for (const auto& det : detector_rects_focal_) {
    auto overlap = det.intersection(illuminated_rect);
    if (!overlap) {
      continue;
    }

    const auto area = overlap->area();
    if (area <= 0.0) {
      continue;
    }

    total_area += area;
    overlap_rects_cache_.push_back(*overlap);
    overlap_area_cdf_cache_.push_back(total_area);
  }

  if (total_area <= 0.0 || overlap_rects_cache_.empty()) {
    healog(5) << "CodedMask reroll: illuminated mask rectangle does not overlap any detector area."
              << " (theta/phi dependent? vignetting might already be ~0)" << std::endl;
    return std::nullopt;
  }

  // Reroll mask sampling (impact position) until we hit an open mask pixel.
  for (size_t attempt = 0; attempt < max_reroll_; ++attempt) {
    const auto r = getUniformRandomNumber() * total_area;
    const auto it = std::lower_bound(overlap_area_cdf_cache_.begin(),
                                     overlap_area_cdf_cache_.end(), r);
    const auto idx = static_cast<size_t>(
        std::distance(overlap_area_cdf_cache_.begin(), it));
    const auto& rect = overlap_rects_cache_.at(idx);

    const auto x = rect.xmin() + getUniformRandomNumber() * (rect.xmax() - rect.xmin());
    const auto y = rect.ymin() + getUniformRandomNumber() * (rect.ymax() - rect.ymin());

    const auto mask_x = x - shift_x;
    const auto mask_y = y - shift_y;
    if (coded_mask_.isOpenAtPhysical(mask_x, mask_y)) {
      return SixtePoint{x, y, 0.0};
    }
  }

  healog(5) << "CodedMask reroll: reached max_reroll=" << max_reroll_
            << " without finding an open mask pixel in overlap region." << std::endl;
  return std::nullopt;
}

RaytracingImagingStrategy::RaytracingImagingStrategy(XMLData& xml_data)
    : mirror_module_(RaytracingImagingStrategy::create_telescope(xml_data)),
      reroll_policy_(readValidatedRerollPolicy(xml_data, ImagingType::RAYTRACER)),
      fov_(xml_data.child("telescope").child("fov").attributeAsDouble("diameter_m")) {
  if (reroll_policy_ != RayRerollPolicy::NONE &&
      dynamic_cast<LobsterEyeOptic*>(mirror_module_.get()) == nullptr) {
    throw SixteException("reroll_policy is only supported for lobster_eye optics");
  }
}


std::unique_ptr<MirrorModule> RaytracingImagingStrategy::create_telescope(XMLData& xml_data) {
  auto raytracing = xml_data.child("telescope").child("raytracer");

  auto mirror_type = raytracing.child("type").attributeAsString("type");
  if (mirror_type == "wolter")
    return std::make_unique<Wolter>(xml_data);
  if (mirror_type == "lobster_eye") {
    return std::make_unique<LobsterEyeOptic>(xml_data);
  }
  throw std::runtime_error("Unknown mirror_module type: " + mirror_type);
}

double generateRandomDouble(double m, double n) {
  double uniform_number = sixte::getUniformRandomNumber();
  return m + (n-m) * uniform_number;
}

bool RaytracingImagingStrategy::shouldReroll(RayFailReason reason) const {
  if (reason == RayFailReason::UNSET) {
    throw SixteException("RayFailReason UNSET in shouldReroll");
  }
  if (reason == RayFailReason::NONE || reason == RayFailReason::TOO_SHALLOW) {
    return false;
  }
  if (reroll_policy_ == RayRerollPolicy::ALL_FAILURES) {
    return true;
  }
  if (reroll_policy_ == RayRerollPolicy::OPTICS_ONLY) {
    return (reason == RayFailReason::MISSED_OPTIC ||
            reason == RayFailReason::LOST_IN_PORE ||
            reason == RayFailReason::MAX_DEPTH);
  }
  return false;
}


std::optional<Ray> RaytracingImagingStrategy::RayTraceReroll(Vec3fa& photon_direction,
                                                             const sixte::SixtePhoton &photon) {
  const long max_reroll_attempts = 10000;

  for (long reroll_counter = 0; reroll_counter < max_reroll_attempts; ++reroll_counter) {
    if (reroll_policy_ == RayRerollPolicy::OPTICS_ONLY) {
      // -----------------------------
      // Stage-1: optics-only reroll
      // -----------------------------
      RayTraceResult optics_result = RayTraceOpticsOnce(photon_direction, photon);
      if (!optics_result.ray.has_value()) {
        if (!shouldReroll(optics_result.fail_reason)) return std::nullopt;
        continue;
      }

      // -------------------------------------------
      // Stage-2: terminal checks (no reroll)
      // -------------------------------------------
      Ray ray_after_optics = *optics_result.ray;
      std::optional<Ray> terminal_result = mirror_module_->ray_trace_terminal(ray_after_optics);
      if (terminal_result.has_value()) {
        terminal_result->set_fail_reason(RayFailReason::NONE);
        return terminal_result;
      }

      // Physical terminal loss (spider/gaps/...): drop photon.
      return std::nullopt;
    }

      if (reroll_policy_ == RayRerollPolicy::ALL_FAILURES) {
      // Full trace + reroll policy applied to the full result.
      RayTraceResult trace_result = RayTraceFullOnce(photon_direction, photon);
      if (trace_result.ray.has_value()) return trace_result.ray;
      if (!shouldReroll(trace_result.fail_reason)) return std::nullopt;
    } else {
      throw SixteException("Unexpected reroll_policy in RayTraceReroll");
    }
  }

  return std::nullopt;
}

RaytracingImagingStrategy::RayTraceResult
RaytracingImagingStrategy::LaunchRay(Vec3fa& photon_direction,
                                     const sixte::SixtePhoton &photon) {
  RayTraceResult trace_result;
  std::pair<double, double> xy = GetRandomPhotonPosInFOV();
  auto entrance_plane_z = static_cast<float>(mirror_module_->entrance_plane_z());
  Vec3fa position{static_cast<float>(xy.first),
                  static_cast<float>(xy.second),
                  entrance_plane_z};

  // Launch ahead of the optics (spider, etc.) while ensuring we still hit the
  // sampled entrance plane position.
  float min_dir_z = 1e-6;
  if (photon_direction.z < -min_dir_z) {
    float launch_offset_mm = 1000;
    const float offset_distance = launch_offset_mm / -photon_direction.z;
    // Offset x/y consistently with the z advance.
    position.x -= photon_direction.x * offset_distance;
    position.y -= photon_direction.y * offset_distance;
    position.z = entrance_plane_z + launch_offset_mm;
  } else {
    // Photon direction is too shallow
    trace_result.fail_reason = RayFailReason::TOO_SHALLOW;
    return trace_result;
  }

  Ray ray = {position, photon_direction, photon.energy()*1000};
  trace_result.ray = ray;
  return trace_result;
}

RaytracingImagingStrategy::RayTraceResult
RaytracingImagingStrategy::RayTraceOpticsOnce(Vec3fa& photon_direction,
                                              const sixte::SixtePhoton &photon) {
  RayTraceResult launch = LaunchRay(photon_direction, photon);
  if (!launch.ray.has_value()) return launch;

  Ray ray = *launch.ray;
  std::optional<Ray> result = mirror_module_->ray_trace_optics(ray);
  if (result.has_value()) {
    result->set_fail_reason(RayFailReason::NONE);
    return RayTraceResult{result, RayFailReason::NONE};
  }

  return RayTraceResult{std::nullopt, ray.fail_reason()};
}

RaytracingImagingStrategy::RayTraceResult
RaytracingImagingStrategy::RayTraceFullOnce(Vec3fa& photon_direction,
                                            const sixte::SixtePhoton &photon) {
  RayTraceResult launch = LaunchRay(photon_direction, photon);
  if (!launch.ray.has_value()) return launch;

  Ray ray = *launch.ray;
  std::optional<Ray> result = mirror_module_->ray_trace(ray);
  if (result.has_value()) {
    result->set_fail_reason(RayFailReason::NONE);
    return RayTraceResult{result, RayFailReason::NONE};
  }

  return RayTraceResult{std::nullopt, ray.fail_reason()};
}

std::pair<double, double> RaytracingImagingStrategy::GetRandomPhotonPosInFOV() const {
  double half_extent = 0.5 * fov_ * 1.1 * 1000;
  double min = -half_extent;
  double max = half_extent;

  double x = generateRandomDouble(min, max);
  double y = generateRandomDouble(min, max);

  return std::make_pair(x, y);
}

std::optional<SixtePhoton> RaytracingImagingStrategy::operator()(sixte::NewAttitude &attitude,
                                                                 const sixte::SixtePhoton &photon) {

  auto telescope = attitude.getTelescopeAxes(photon.time());
  auto source_direction = unitVector(*photon.ra(), *photon.dec());

  Vec3fa photon_direction = {
    float(scalarProduct(source_direction, telescope.nx)),
    float(scalarProduct(source_direction, telescope.ny)),
    float(-scalarProduct(source_direction, telescope.nz))
  };

  photon_direction = normalize(photon_direction); //TODO: Check if needed

  std::optional<Ray> result;

  if (reroll_policy_ == RayRerollPolicy::NONE) {
    auto trace_result = RayTraceFullOnce(photon_direction, photon);
    result = trace_result.ray;
  } else {
    result = RayTraceReroll(photon_direction, photon);
  }

  if (!result.has_value()) return std::nullopt;

  result.value().sensor_position = result.value().sensor_position / 1000; // Conversion from mm to m

  auto hit = result.value().position();
  auto metainfo = PhotonMetainfo(photon.photon_metainfo().ph_id_,
                                 photon.photon_metainfo().src_id_,
                                 photon.photon_metainfo().tel_id_,
                                 photon.photon_metainfo().chip_id_,
                                 SixteVector(hit.x, hit.y, hit.z));

  auto out_photon = SixtePhoton(photon.time(),
                                photon.energy(),
                                SixtePoint(result.value().sensor_position.x,
                                           result.value().sensor_position.y,
                                           0),
                                metainfo);
  out_photon.setChipID(result.value().hitID);

  return out_photon;
}

ImagingType readImagingType(XMLData& xml_data) {
  auto telescope = xml_data.child("telescope");
  if (telescope.hasAttribute("imaging")) {
    std::string imaging_type = telescope.attributeAsString("imaging");

    if (imaging_type == "psf") return ImagingType::PSF;
    if (imaging_type == "codedmask") return ImagingType::CODEDMASK;
    if (imaging_type == "raytracer") return ImagingType::RAYTRACER;

    throw SixteException("Unknown imaging type '" + imaging_type + "'");
  }

  healog(5) << "Imaging attribute not set in XML, defaulting to psf" << std::endl;
  return ImagingType::PSF;
}

size_t readMaxReroll(XMLData& xml_data) {
  auto telescope = xml_data.child("telescope");
  if (!telescope.hasChild("arf")) return 10000;

  auto arf = telescope.child("arf");
  if (!arf.hasAttribute("max_reroll")) return 10000;

  const auto max_reroll = arf.attributeAsInt("max_reroll");
  if (max_reroll <= 0) {
    throw SixteException("max_reroll must be > 0, got " + std::to_string(max_reroll));
  }
  return static_cast<size_t>(max_reroll);
}

PhotonImaging::PhotonImaging(const std::string& output_file, bool clobber,
                             XMLData& xml_data, const ObsInfo& obs_info)
    : imaging_strategy_(createImagingStrategy(xml_data)) {
  if (!output_file.empty()) {
    write_impact_file_ = true;
    impact_file_.emplace(output_file, clobber, obs_info);
  }
}

std::optional<SixtePhoton> PhotonImaging::doImaging(NewAttitude& attitude,
                                                    const SixtePhoton& photon) {
  auto isimg = imaging_strategy_(attitude, photon);
  if (isimg) {
    num_imaged_++;
    if (write_impact_file_) impact_file_->addImpact2File(*isimg);
  }
  return isimg;
}

void PhotonImaging::checkIfImaged() const {
  if (num_imaged_ == 0) {
    printWarning(
        "No photons imaged by the telescope! Check your pointing or exposure "
        "time");
  }
}

ImagingStrategy PhotonImaging::createImagingStrategy(XMLData& xml_data) {
  auto imaging_type = readImagingType(xml_data);

  switch (imaging_type) {
    case ImagingType::PSF:
      return PSFImagingStrategy(xml_data);

    case ImagingType::CODEDMASK:
      return CodedMaskImagingStrategy(xml_data);

    case ImagingType::RAYTRACER:
      return RaytracingImagingStrategy(xml_data);

    default:
      throw SixteException("Unknown imaging type");
  }
}

bool isPhotonInFov(const SixtePhoton& photon, NewAttitude& attitude, double fov_diameter) {
  const double fov_min_align = cos(fov_diameter / 2.);
  auto photon_direction = unitVector(*photon.ra(), *photon.dec());
  auto telescope_nz = attitude.getTelescopeNz(photon.time());
  return isInFov(photon_direction, telescope_nz, fov_min_align);
}
}  // namespace sixte
