/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/
#include "PhotonGeneration.h"

#include "NewPhotonfile.h"

namespace sixte {

PhotonGeneration::PhotonGeneration(const std::vector<std::string>& photon_lists, bool clobber,
                                   const std::vector<std::string>& simput_files,
                                   std::vector<XMLData>& xml_datas,
                                   const std::vector<ObsInfo>& obs_infos)
    : num_tels_{xml_datas.size()}, arf_(getARFFilenames(xml_datas)) {
  
  if (simput_files.empty()){
    printWarning("No Simput files given "
                 "(must provide at least one, otherwise you will only get background)");
  }
  
  size_t idx = getIDofXMLwithLargestFOV(xml_datas);
  double fov = xml_datas[idx].child("telescope").child("fov").attributeAsDouble("diameter");
  fov_diameter_rad_ = fov * M_PI / 180.;
  
  for (const auto& simput_file: simput_files) {
   source_catalog_.emplace_back(simput_file, arf_);
  }
  
  for (size_t ii=0; ii<num_tels_; ii++) {
    if (!photon_lists[ii].empty()) {
      write_photon_file_ = true;
      photon_file_.emplace_back(photon_lists[ii], clobber, obs_infos[0]);
    }
  }
}

std::optional<SixtePhoton> PhotonGeneration::getNextPhoton(NewAttitude& attitude, std::pair<double, double> dt) {
  auto src_photon = phgen(dt, attitude);

  if (src_photon) {
    if (src_photon->time() > dt.second) {
      throw SixteException("Photon time greater than requested time interval");
    }
    
    size_t photon_telid = arf_.getTelescopeID(src_photon->energy());
    src_photon->setTelID(photon_telid);
    
    if (write_photon_file_) photon_file_[photon_telid].addPhoton2File(*src_photon);
  }
  return src_photon;
}

std::optional<SixtePhoton> PhotonGeneration::phgen(std::pair<double, double> dt,
                                                   NewAttitude& attitude) {
  // Photon list buffer.
  static SixtePhotons photon_buffer;
  
  //static LinkedPhoListElement* pholist=NULL;
  // Counter for the photon IDs.
  static long long ph_id = 1;
  
  // Current time.
  static double time = 0.;
  
  if (time<dt.first) time = dt.first;
  
  // If the photon list is empty generate new photons from the
  // given source catalog.
  while (photon_buffer.empty() && (time < dt.second)) {
    // Determine the telescope pointing at the current point of time.
    
    // Generate new photons for all specified catalogs.
    double t1 = std::min(time + attitude.dt(), dt.second);
    
    //ML: for (const auto& source_catalog: source_catalogs_)
    for (const auto& source_catalog: source_catalog_) {
      // Get photons for all sources in the catalog.
      int status = EXIT_SUCCESS;
      LinkedPhoListElement* newlist = source_catalog.sixteGenFoVXRayPhotons(attitude, fov_diameter_rad_, time, t1);
      checkStatusThrow(status, "Failed to generate Photon List");
      
      while (newlist) {
        newlist->photon.ph_id = ph_id++;
        photon_buffer.emplace(newlist->photon);
        newlist = newlist->next;
      }
    }
    time += attitude.dt();
  }
  if (photon_buffer.empty()) return(std::nullopt);
  return pop_next_photon(photon_buffer);
}

std::vector<std::string> getARFFilenames (std::vector<XMLData>& xml_datas) {
  std::vector<std::string> arf_filenames;
  for (auto& xml_data : xml_datas) {
    std::string arf_filename = xml_data.dirname()
      + xml_data.child("telescope").child("arf").attributeAsString("filename");
    if (arf_filename.empty()) throw SixteException("ARF filename not found");
    arf_filenames.push_back(arf_filename);
  }
  return arf_filenames;
}

size_t getIDofXMLwithLargestFOV(std::vector<XMLData>& xml_datas) {
  double largest_fov_diameter = 0.0;  // [radians]
  std::vector<double> fov_record;
  
  size_t index = 0;
  for (size_t telid = 0; telid < xml_datas.size(); telid++) {
    double fov_diameter_rad = xml_datas[telid].child("telescope").child("fov").attributeAsDouble("diameter") * M_PI / 180.;
    fov_record.push_back(fov_diameter_rad);
    if (fov_diameter_rad > largest_fov_diameter){
      largest_fov_diameter = fov_diameter_rad;
      index = telid;
    }
  }
  
  if (not std::equal(fov_record.begin() + 1, fov_record.end(), fov_record.begin())) {
    healog(5) << "Field of views are not the same, using largest one of telescope " << index
              << " with value " << largest_fov_diameter << "rad" << std::endl;
  }
  
  return index;
}

}
