/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "Parameters.h"

#include "healog.h"
#include "NewSIXT.h"
#include "SixteCCFits.h"
#include "XMLData.h"

#include <utility>
#include <iostream>
#include <vector>

namespace sixte {

bool isNone(const std::string& parameter_name) {
  return boost::iequals(parameter_name, "none");
}

std::vector<std::string> loadSIMPUTFilenames(){
  std::vector<std::string> simput_files;
  std::string simput_file_list = queryParameterString("Simput");
  
  simput_files = separate_fits_string(simput_file_list, ',');
  
  healog(5) << "using simput files: " << std::endl;
  for (const std::string& simput_file:simput_files) healog(5) << simput_file << std::endl;

  if (simput_files.empty()) {
    throw SixteException("At least one SIMPUT File must be given, did you gave a file for \"Simput\"? ");
  }
  return simput_files;
}

XMLDetectorFiles loadXMLFiles() {
  std::vector<std::string> xml_file_names;
  std::string xml_file_list = queryParameterString("XMLFile");
  
  xml_file_names = separate_string_to_strings(xml_file_list, ',');
  
  if (xml_file_names.empty()) {
    throw SixteException("At least one XML File must be given, did you gave a file for \"XMLFiles\"? ");
  }
  
  healog(5) << "using xml files: " << std::endl;
  for (const std::string& xml_file : xml_file_names) healog(5) << xml_file << std::endl;

  return detectorXMLs(xml_file_names);
}

std::vector<pugi::xml_document> loadTelXMLFiles() {
  std::vector<std::string> xml_file_names;
  std::string xml_file_list = queryParameterString("XMLFile");
  
  xml_file_names = separate_string_to_strings(xml_file_list, ',');
  
  if (xml_file_names.empty()) {
    throw SixteException("At least one XML File must be given, did you gave a file for \"XMLFiles\"? ");
  }
  
  healog(5) << "using xml files: " << std::endl;
  for (const std::string& xml_file : xml_file_names) healog(5) << xml_file << std::endl;
  
  std::vector<pugi::xml_document> xml_docs;
  for (const auto& xml_file : xml_file_names) {
    pugi::xml_document xml_document;
    xml_document.load_file(xml_file.c_str());
    xml_docs.emplace_back(std::move(xml_document));
  }
  return xml_docs;
}

std::string loadXMLPath() {
  std::vector<std::string> xml_files;
  std::string xml_file_list = queryParameterString("XMLFile");
  
  xml_files = separate_string_to_strings(xml_file_list, ',');
  return xml_files.front();
}

size_t getNumTels() {
  std::vector<std::string> xml_files;
  std::string xml_file_list = queryParameterString("XMLFile");
  
  xml_files = separate_string_to_strings(xml_file_list, ',');
  return xml_files.size();
}

InputFiles::InputFiles() {
  XMLDetectorFiles xml_detector_files = loadXMLFiles();
  for (auto& xml_doc : xml_detector_files.xml_documents) {
    xml_files.push_back(std::move(xml_doc));
  }
  xml_tel_files = loadTelXMLFiles();
  xml_path = loadXMLPath();
  simput_files = loadSIMPUTFilenames();
  gti_file = queryParameterString("GTIfile");
  num_xmls = xml_files.size();
  num_tels = getNumTels();
  num_chips = xml_detector_files.num_chips;
}

OutputFiles::OutputFiles(
    std::optional<std::string> o_prefix,
    std::optional<std::string> o_photon_list,
    std::optional<std::string> o_impact_list,
    std::optional<std::string> o_evt_file,
    std::optional<std::string> o_raw_data,
    std::optional<bool> o_clobber) {

  prefix = o_prefix.has_value() ? o_prefix.value(): queryParameterString("Prefix");
  if (isNone(prefix)) {
    prefix.clear();
  }

  std::string photon_list = o_photon_list.has_value() ? o_photon_list.value(): queryParameterString("PhotonList");
  std::string impact_list = o_impact_list.has_value() ? o_impact_list.value(): queryParameterString("ImpactList");
  std::string evt_file = o_evt_file.has_value() ? o_evt_file.value(): queryParameterString("EvtFile");
  std::string raw_data = o_raw_data.has_value() ? o_raw_data.value(): queryParameterString("RawData");
  clobber = o_clobber.has_value() ? o_clobber.value(): queryParameterBool("clobber");

  if (raw_data.empty() || isNone(raw_data)) {
    delete_rawdata = true;
    raw_data = "raw.fits";
  } else {
    delete_rawdata = false;
  }

  std::vector<std::string> xml_file_names = separate_string_to_strings(queryParameterString("XMLFile"), ',');
  
  // handle the impact and photon lists (one per telescope)
  for (const auto& xml_file_name : xml_file_names) {
    pugi::xml_document xml_file;
    xml_file.load_file(xml_file_name.c_str());
    XMLData xml_data(xml_file, "");
    
    auto string_prefix = prefix;
    
    if (xml_file_names.size() > 1) {
      try {
        std::string tel_name = xml_data.child("telescope").attributeAsString("number");
        string_prefix += "tel" + tel_name + "_";
      } catch(AttributeNotFound &e) {
          throw SixteException("Telescope numbers have to be given in xml to simulate multiple telescopes!");
      }
    }
    
    if (!isNone(photon_list)) photon_lists.push_back(string_prefix + photon_list);
    else photon_lists.emplace_back("");
    if (!isNone(impact_list)) impact_lists.push_back(string_prefix + impact_list);
    else impact_lists.emplace_back("");
  }

  auto xml_files = detectorXMLs(xml_file_names).xml_documents;

  // handle the event files (one per chip)
  for (const auto& xml_file : xml_files) {
    XMLData xml_data(xml_file, "");

    auto string_prefix = prefix;

    if (xml_file_names.size() > 1) {
      try {
        std::string tel_name = xml_data.child("telescope").attributeAsString("number");
        string_prefix += "tel" + tel_name + "_";
      } catch(AttributeNotFound &e) {
          throw SixteException("Telescope numbers have to be given in xml to simulate multiple telescopes!");
      }
    }

    if (xml_file_names.size() < xml_files.size()) {
      try {
        int chip_name = xml_data.child("detector").attributeAsInt("chip");
        string_prefix += "chip" + std::to_string(chip_name) + "_";
      } catch(AttributeNotFound &e) {
          throw SixteException("Chip numbers have to be given in xml to simulate multiple chips!");
      }
    }

    evt_files.push_back(string_prefix + evt_file);
    raw_datas.push_back(string_prefix + raw_data);
  }
}

ObsPointing::ObsPointing() {
  attitude_file = queryParameterString("Attitude");
  if (isNone(attitude_file)) {
    ra = queryParameterDouble("RA");
    dec = queryParameterDouble("Dec");
    rollangle = queryParameterDouble("rollangle");
    attitude_file.clear();
  } else {
    ra = 0.;
    dec = 0.;
    rollangle = 0.;
    healog(3) << "using Attiude File: " << attitude_file << '\n';
  }
}

ObsPointing::ObsPointing(std::string attitude_file_, double ra_, double dec_, double rollangle_)
: attitude_file(std::move(attitude_file_)), ra(ra_), dec(dec_), rollangle(rollangle_) {}


void ObsPointing::addPointingInfo(CCfits::ExtHDU& ext_hdu) const {
  if (attitude_file.empty()) {
    ext_hdu.addKey("RA_PNT", ra, "RA of pointing direction [deg]", false);
    ext_hdu.addKey("DEC_PNT", dec, "Dec of pointing direction [deg]",false);
    ext_hdu.addKey("PA_PNT", rollangle, "Roll angle [deg]", false);
  } else {
    ext_hdu.addKey("ATTITUDE", attitude_file, "attitude file", false);
  }
}

void ObsPointing::addPointingInfo(Fitsfile& fitsfile) const {
  if (attitude_file.empty()) {
    fitsfile.updateKey("RA_PNT", ra, "RA of pointing direction [deg]");
    fitsfile.updateKey("DEC_PNT", dec, "Dec of pointing direction [deg]");
    fitsfile.updateKey("PA_PNT", rollangle, "Roll angle [deg]");
  } else {
    fitsfile.updateKey("ATTITUDE", attitude_file, "attitude file");
  }
}

void ObsPointing::addPointingInfo(fitsfile *fptr) const {
  if (attitude_file.empty()) {
    cfitsio::fitsUpdateKey(fptr, "RA_PNT", &ra, "RA of pointing direction [deg]");
    cfitsio::fitsUpdateKey(fptr, "DEC_PNT", &dec, "Dec of pointing direction [deg]");
    cfitsio::fitsUpdateKey(fptr, "PA_PNT", &rollangle, "Roll angle [deg]");
  } else {
    cfitsio::fitsUpdateKeyLongstr(fptr, "ATTITUDE", attitude_file, "attitude file");
  }
}

ObsTime::ObsTime() {
  mjdref = queryParameterDouble("MJDREF");
  tstart = queryParameterDouble("TSTART");
  exposure = queryParameterDouble("Exposure");
}

ObsTime::ObsTime(double mjdref_, double tstart_, double exposure_)
: mjdref(mjdref_), tstart(tstart_), exposure(exposure_) {}


XMLDetectorFiles detectorXMLs(const std::vector<std::string>& xml_filenames) {
  XMLDetectorFiles xml_detector_files;

  for (auto& xml_filename : xml_filenames) {
    XMLData xml_data(xml_filename);
    auto& xml_file = xml_data.document();
    
    auto old_instrument_node = xml_file.child("instrument");
    auto telescope_node = old_instrument_node.child("telescope");
    
    size_t numChips = 0;
    
    for (pugi::xml_node detector_node : old_instrument_node.children("detector")) {
      pugi::xml_document new_document;
      
      pugi::xml_node new_instrument_node = new_document.append_child("instrument");
      for (pugi::xml_attribute attribute : old_instrument_node.attributes()) {
        new_instrument_node.append_copy(attribute);
      }
      
      new_instrument_node.append_copy(telescope_node);
      new_instrument_node.append_copy(detector_node);
      
      xml_detector_files.xml_documents.push_back(std::move(new_document));
      numChips ++;
    }
    xml_detector_files.num_chips.emplace_back(numChips);
  }
  
  return xml_detector_files;
}

}


