/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <memory>
#include <vector>

#include "Ebounds.h"
#include "PatternAnalysis.h"
#include "Pha2PiCorrection.h"
#include "PhotonProjection.h"
#include "Postprocessing.h"
#include "Registry.h"
#include "ReadoutClock.h"
#include "ReadoutStrategy.h"
#include "SimulationParameters.h"

namespace sixte {

class ShiftArray;
class PixArray;
class Absorber;

class TimeTriggeredOperation {
 public:
  TimeTriggeredOperation(XMLData& xml_data, const OutputFiles& outfiles,
                         const ObsInfo& obs_info, bool skip_invalids, size_t xml_id,
                         const std::shared_ptr<const Ebounds>& ebounds);

  void propagate(ShiftArray& sensor, double tstop);

  void postprocessing(ShiftArray& sensor, Absorber& absorber,
                      NewAttitude& attitude, const GTICollection& gti);

  void setReadoutStrategy(std::unique_ptr<ShiftArrayReadoutStrategy>&& strategy);

  double frameDuration() {
    return readout_clock_.frameDuration();
  }

  void setStartTime(double tstart);

 private:
  ReadoutClock readout_clock_;

  NewEventfile event_file_;
  std::optional<PatternAnalysis> pattern_analysis_;
  std::optional<PhotonProjection> photon_projection_;
  std::optional<Pha2PiCorrection> pha2pi_correction_;
  double focal_length_; // telescope focal length
  PostprocessingConfiguration postprocessing_config_;
};


class EventTriggeredOperation {
 public:
 EventTriggeredOperation(std::string eventfile_name, bool clobber, bool skip_invalids, XMLData& xml_data, const ObsInfo& obs_info,
                         const std::shared_ptr<const Ebounds>& ebounds);

  void propagate(PixArray& sensor, double tstop);

  void postprocessing(PixArray& sensor, Absorber& absorber,
                      NewAttitude& attitude, const GTICollection& gti);

  void setReadoutStrategy(std::unique_ptr<PixArrayReadoutStrategy>&& strategy);

 private:
  std::unique_ptr<PixArrayReadoutStrategy> readout_strategy_;
  double focal_length_; // telescope focal length

  std::optional<PatternAnalysis> pattern_analysis_;
  std::optional<PhotonProjection> photon_projection_;
  std::optional<Pha2PiCorrection> pha2pi_correction_;
  NewEventfile event_file_;
  PostprocessingConfiguration postprocessing_config_;
};

class MicroCalOperation {
 public:
  MicroCalOperation(XMLData& xml_data,
      const std::string& event_file,
      bool clobber,
      const ObsInfo& obs_info,
      std::shared_ptr<RmfRegistry> rmf_registry);

  void propagate(MicroCal& sensor, double tstop);

  void postprocessing(MicroCal& sensor, Absorber& absorber,
    NewAttitude& attitude, const GTICollection& gti);

  double t_samp() {return t_samp_;}

  void runReadout(MicroCal& sensor, T_PixId id, std::optional<double> t_read);

 private:
  MicroCalReadoutStrategy readout_strategy_;

  double t_samp_; // sample time
  double max_n_post_; // maximum post record length
  double focal_length_; // telescope focal length
  double readout_check_interval_; // intervals in which to check if pixels can be read out
  std::optional<double> last_readout_check_ {std::nullopt};
  PostprocessingConfiguration postprocessing_config_;
};


}
