/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "OperationStrategy.h"
#include "Sensor.h"
#include "XMLData.h"
#include "healog.h"
#include "Absorber.h"
#include "Postprocessing.h"
#include <optional>

namespace sixte {

TimeTriggeredOperation::TimeTriggeredOperation(XMLData& xml_data, const OutputFiles& outfiles, const ObsInfo& obs_info, bool skip_invalids, size_t xml_id,
                                               const std::shared_ptr<const Ebounds>& ebounds)
    : readout_clock_(xml_data),
      event_file_(outfiles.evt_files[xml_id], outfiles.clobber, xml_data, obs_info,
                  static_cast<size_t>(ebounds->firstChannel()),
                  ebounds->numberChannels()),
      postprocessing_config_(xml_data) {
  if (postprocessing_config_.needsPatternAnalysis()) {
    pattern_analysis_.emplace(skip_invalids,
                              xml_data,
                              readout_clock_.rawYMin(),
                              readout_clock_.rawYMax());
  }

  if (postprocessing_config_.needsPhotonProjection()) {
    photon_projection_.emplace(xml_data);
  }

  if (postprocessing_config_.needsPha2PiCorrection()) {
    pha2pi_correction_.emplace(xml_data);
  }

  focal_length_ = xml_data.child("telescope").child("focallength").attributeAsDouble("value");
}

void TimeTriggeredOperation::propagate(ShiftArray& sensor, double tstop) {
  readout_clock_.operate(sensor, tstop);
}

void TimeTriggeredOperation::postprocessing(
    ShiftArray& sensor, Absorber& absorber,
    NewAttitude& attitude, const GTICollection& gti) {

  processPatternAnalysis(pattern_analysis_, readout_clock_.ebounds(),
                         readout_clock_.rawEventFile(), event_file_);

  processPhotonProjection(photon_projection_, attitude, event_file_,
                          gti.tstartFirstGti(), gti.tstopLastGti());

  processPha2PiCorrection(pha2pi_correction_, event_file_);

  processFocalPlaneCoordinatesRectangular(postprocessing_config_.needsFocalPlaneCoordinates(),
                                          event_file_, *sensor.rectangularArrayGeometry(),
                                          *absorber.geometry(), absorber.chipId());

  processGtiExtension(gti, event_file_.fitsfile().rawPtr());
}

void TimeTriggeredOperation::setReadoutStrategy(std::unique_ptr<ShiftArrayReadoutStrategy>&& strategy) {
  readout_clock_.setReadoutStrategy(std::move(strategy));
}

void TimeTriggeredOperation::setStartTime(double tstart) {
  readout_clock_.setStartTime(tstart);
}

EventTriggeredOperation::EventTriggeredOperation(std::string eventfile_name, bool clobber, bool skip_invalids, XMLData& xml_data, const ObsInfo& obs_info,
                                                 const std::shared_ptr<const Ebounds>& ebounds)
      : event_file_(eventfile_name, clobber, xml_data, obs_info,
                    static_cast<size_t>(ebounds->firstChannel()),
                    ebounds->numberChannels()),
        postprocessing_config_(xml_data) {
  if (postprocessing_config_.needsPatternAnalysis()) {
    auto detector = xml_data.child("detector");
    auto parent_node = detector.optionalChild("geometry").value_or(detector);
    auto dimensions = parent_node.child("dimensions");

    size_t rawymin = 0;
    size_t rawymax = dimensions.attributeAsInt("ywidth") - 1;
    pattern_analysis_.emplace(skip_invalids,
                              xml_data,
                              rawymin,
                              rawymax);
  }

  if (postprocessing_config_.needsPhotonProjection()) {
    photon_projection_.emplace(xml_data);
  }

  if (postprocessing_config_.needsPha2PiCorrection()) {
    pha2pi_correction_.emplace(xml_data);
  }

  focal_length_ = xml_data.child("telescope").child("focallength").attributeAsDouble("value");
}

void EventTriggeredOperation::propagate(PixArray& sensor, double /*tstop*/) {
  if (!sensor.anySignal()) return;

  // Read out all active pixels and clear them afterwards.
  auto active_pixels = sensor.getActiveInArray();
  readout_strategy_->readoutPixels(sensor, active_pixels);
}

void EventTriggeredOperation::setReadoutStrategy(std::unique_ptr<PixArrayReadoutStrategy>&& strategy) {
  readout_strategy_ = std::move(strategy);
}
void EventTriggeredOperation::postprocessing(
    PixArray& sensor, Absorber& absorber,
    NewAttitude& attitude, const GTICollection& gti) {

  processPatternAnalysis(pattern_analysis_, readout_strategy_->ebounds(),
                         readout_strategy_->rawEventFile(), event_file_);

  processPhotonProjection(photon_projection_, attitude, event_file_,
                          gti.tstartFirstGti(), gti.tstopLastGti());

  processPha2PiCorrection(pha2pi_correction_, event_file_);

  processFocalPlaneCoordinatesRectangular(postprocessing_config_.needsFocalPlaneCoordinates(),
                                          event_file_, *sensor.rectangularArrayGeometry(),
                                          *absorber.geometry(), absorber.chipId());

  processGtiExtension(gti, event_file_.fitsfile().rawPtr());
}

MicroCalOperation::MicroCalOperation(
    XMLData& xml_data,
    const std::string& event_file, bool clobber,
    const ObsInfo& obs_info,
    std::shared_ptr<RmfRegistry> rmf_registry)
: readout_strategy_(xml_data, clobber, event_file, obs_info, std::move(rmf_registry)),
  postprocessing_config_(xml_data)
{
  max_n_post_ = readout_strategy_.max_n_post();
  t_samp_ = readout_strategy_.t_samp();
  readout_check_interval_ = 2 * max_n_post_ * t_samp_;
  focal_length_ = xml_data.child("telescope").child("focallength").attributeAsDouble("value");
}

void MicroCalOperation::propagate(MicroCal& sensor, double tstop) {
  if (!sensor.anySignal()) {
    return;
  }

  // check if any of the active pixels can be read out
  // but only if the last check has not been too recent
  // (this is an optimization for high count rates)
  if (!last_readout_check_.has_value()
      || (tstop - last_readout_check_.value() > readout_check_interval_)) {

    auto& active_pixels = sensor.getActivePixels();

    // copy into a vector to iterate over - active_pixels can change due to readout!
    // (copying to vector is faster than copying a set)
    std::vector<T_PixId> ids(active_pixels.begin(), active_pixels.end());

    for (auto& id: ids) {
      auto t_sig = sensor.getSignal(id)->creation_time();
      if (tstop - t_sig > max_n_post_*t_samp_) {
        runReadout(sensor, id, std::nullopt);
      }

      last_readout_check_ = tstop;
    }
  }
}

void MicroCalOperation::postprocessing(
    MicroCal& sensor,
    Absorber& absorber,
    NewAttitude& attitude,
    const GTICollection& gti) {

  // TODO should not be needed once doPhotonProjection uses CCFits
  int status=EXIT_SUCCESS;
  fits_flush_file(readout_strategy_.eventFile()->fptr, &status);
  checkStatusThrow(status, "Failed to flush FITS file");

  if (postprocessing_config_.needsPhotonProjection()) {
    healog(3) << "start sky projection ...\n";
    doMicroCalPhotonProjection(attitude, *absorber.geometry(),
                                                  *sensor.arrayGeometry(),
                                                  focal_length_,
                                                  readout_strategy_.eventFile()->fptr,
                                                  gti.tstartFirstGti(), gti.tstopLastGti());
  }

  gti.saveGtiExtension(readout_strategy_.eventFile()->fptr);
}

void MicroCalOperation::runReadout(
    MicroCal& sensor, T_PixId id,
    std::optional<double> t_read
    ) {
  const auto& sig = sensor.getSignal(id);

  if (!sig.has_value()) {
    std::cout
      << "Tried to read out pixel " << id << " without signal"
      << std::endl;
    return;
  }

  std::optional<unsigned int> n_pre;
  std::optional<unsigned int> n_post;
  auto t_sig = sig->creation_time();

  // pre samples
  auto last_readout = sensor.getLastReadoutTime(id);
  if (last_readout.has_value()) {
    n_pre = static_cast<unsigned int>
      ((t_sig - last_readout.value()) / t_samp_);
  }

  // post_samples
  if (t_read.has_value()) {
    n_post = static_cast<unsigned int>
      ((t_read.value() - t_sig) / t_samp_);
  }

  readout_strategy_.readoutPixel(sensor, id, n_pre, n_post);
}

}
