/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "ObsInfo.h"
#include <chrono>

namespace sixte {

ObsInfo::ObsInfo(XMLData& xml_data, ObsPointing pointing_in, const GTICollection& gti)
  : pointing(std::move(pointing_in)) {
  telescop = xml_data.root().attributeAsString("telescop");
  instrume = xml_data.root().attributeAsString("instrume");
  ancrfile = xml_data.child("telescope").child("arf").attributeAsString("filename");
  if (auto rmf = xml_data.child("detector").optionalChild("rmf")) {
    respfile = rmf->attributeAsString("filename");
  } else {
    respfile = "NONE";
  }
  
  CCfits::FITS arfFile(xml_data.dirname() + ancrfile, CCfits::Read, "SPECRESP");
  try {
  arfFile.extension("SPECRESP").readKey("FILTER", filter);
  } catch (const CCfits::HDU::NoSuchKeyword &e) {
  filter = "NONE";
  }
  
  mjdref = gti.mjdref();
  tstart = gti.tstartFirstGti();
  tstop = gti.tstopLastGti();
  timezero = gti.timezero();
  total_sim_time = gti.totalExposure();
}

void ObsInfo::addObsInfoKeysToFile(CCfits::HDU &table) const {
  
  // Set the mission keywords.
  table.addKey("TELESCOP", telescop, "", false);
  table.addKey("INSTRUME", instrume, "", false);
  table.addKey("FILTER", filter, "", false);
  
  // Set the names of the response files used for the simulation.
  table.addKey("ANCRFILE", ancrfile, "ancillary response file", false);
  table.addKey("RESPFILE", respfile, "response file", false);
  
  auto current_time = std::chrono::system_clock::now();
  
  // Convert the time point to a time_t
  std::time_t currentTime_t = std::chrono::system_clock::to_time_t(current_time);
  
  // Convert the time_t to a tm structure
  std::tm *timeInfo = std::gmtime(&currentTime_t);
  
  // Formatting time into a string
  char buffer[1000];  // Buffer to hold the formatted time
  std::strftime(buffer, sizeof(buffer), "%Y-%m-%dT%T", timeInfo); // Formatting time into the buffer
  std::string current_datetimestr(buffer); // Creating a string from the formatted buffer
  
  if (!current_datetimestr.empty()) table.addKey("DATE", current_datetimestr, "file creation date", false);
  // Determine the start date and time.
  std::pair<string, string> date_time_string = sixtGetDateTime(mjdref, tstart);
  
  table.addKey("DATE-OBS", date_time_string.first, "UT date of observation start", false);
  table.addKey("TIME-OBS", date_time_string.second, "UT time of observation start", false);
  
  // Determine the stop date and time.
  date_time_string = sixtGetDateTime(mjdref, tstop);
  table.addKey("DATE-END", date_time_string.first, "UT date of observation end", false);
  table.addKey("TIME-END", date_time_string.second, "UT time of observation end", false);
  table.addKey("MJDREF", mjdref, "reference MJD", false);
  table.addKey("TIMEZERO", timezero, "time offset", false);
  table.addKey("TSTART", tstart, "start time", false);
  table.addKey("TSTOP", tstop, "stop time", false);
  
  // Add header information about program parameters if this is the primary extension.
  if (table.index() == 0) {
    int status = EXIT_SUCCESS;
    HDpar_stamp(table.fitsPointer(), 1, &status);
    checkStatusThrow(status, "Adding runtime information to impact file failed!");
  }
  table.writeChecksum();
}

}