/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "SixteCCFits.h"
#include <string>
#include <vector>

namespace sixte {

class NewVignetting {
 public:
  /** Constructor of the Vignetting data structure. Loads the
    vignetting function from a given FITS file. The format of the
    FITS file is defined by OGIP Memo CAL/GEN/92-021. */
  explicit NewVignetting(const std::string& filename);
  
  /** Determine the Vignetting factor for given photon energy, off-axis
  angle, and azimuth angle. The energy has to be given in [keV], the
  angles in [rad]. If the pointer to the Vignetting data structure
  is NULL, a default value of 1. will be returned. */
  double getVignettingFactor(double psf_energy, double psf_theta, double psf_phi);
  
  [[nodiscard]] const std::vector<double>& energy () const {
    return energy_;
  }
  [[nodiscard]] const std::vector<double>& theta () const {
    return theta_;
  }
  [[nodiscard]] const std::vector<double>& phi () const {
    return phi_;
  }
  [[nodiscard]] const std::pair<double, double> Emin_Emax () const {
    return std::make_pair(Emin_, Emax_);
  }
  
  [[nodiscard]] const std::vector<std::vector<std::vector<double>>>& vignetting_array () const {
    return vignetting_array_;
  }
 
 private:
  double interpolateVigThetaPhi(double psf_theta,
                                double psf_phi,
                                const std::vector<std::vector<double>>& vign_array);
  void checkDimensions (CCfits::ExtHDU& table) const;
  void createVignettingVector (CCfits::ExtHDU& table);
  
  /** Mean energy of bin in [keV] (0.5*(energ_lo+energ_hi). */
  std::vector<double> energy_;
  
  /** Number of energy bins. */
  size_t nenergies_;
  /** Number of off-axis angles. */
  size_t ntheta_;
  /** Number of azimuth angles. */
  size_t nphi_;
  

  /** Off-axis angle in [rad]. */
  std::vector<double> theta_;
  /** Azimuthal angle in [rad]. */
  std::vector<double> phi_;
  /** Vignetting data. Array[energy, theta, phi] */
  std::vector<std::vector<std::vector<double>>> vignetting_array_;
  
  /** Minimum available energy [keV]. */
  double Emin_;
  /** Maximum available energy [keV]. */
  double Emax_;
  
  bool is_vignetting_ = false;
};

}
