/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "NewVignetting.h"

#include "SixteException.h"
#include "healog.h"
#include "FITS.h"
#include "Column.h"
#include "NewSIXT.h"

#include <cmath>

namespace sixte {

NewVignetting::NewVignetting(const std::string& filename) {
  
  if (filename.empty()) throw SixteException("Vignetting file not specified!");
  std::unique_ptr<CCfits::FITS> inFile;
  inFile = sixteOpenFITSFileRead(filename, "Vignetting file");
  is_vignetting_ = true;
  
  auto& table = inFile->extension("VIGNET");
  
  try {
    energy_ = get_columns(table, "ENERGY", 1);
    nenergies_ = energy_.size();
  } catch (const CCfits::Table::NoSuchColumn& e) {
    healog(5) << "Column ENERGY not fount in vignetting file: " << '\n'
              << filename << ". \n"
              << "-> using ENERG_LO and ENERG_HI instead" << '\n';
    
    std::vector<double> energ_lo = get_columns (table, "ENERG_LO", 1);
    std::vector<double> energ_hi = get_columns (table, "ENERG_HI", 1);
    
    if (energ_lo.size() != energ_hi.size()) {
      throw SixteException("Columns ENERG_LO and ENERG_HI do not have the same length!");
    }
    
    for (size_t ii = 0; ii < energ_lo.size(); ii++) {
      energy_.push_back( 0.5 * (energ_lo[ii] + energ_hi[ii]) );
    }
    nenergies_ = energ_lo.size();
  }
  
  // Determine the minimum and maximum available energy:
  Emin_ = energy_[0];
  Emax_ = energy_.back();
  
  theta_ = get_columns (table, "THETA", 1);
  ntheta_ = theta_.size();

  // Try reading PHI as a vector column first; fall back to scalar columns
  try {
    phi_ = get_columns(table, "PHI", 1);
  } catch (const CCfits::Column::WrongColumnType&) {
    table.column("PHI").read(phi_, 1, 1);
  }
  nphi_ = phi_.size();

  if (ntheta_ == 0) {
    throw SixteException("THETA column in vignetting file is empty");
  }
  if (nphi_ == 0) {
    throw SixteException("PHI column in vignetting file is empty");
  }

  // Scale from [deg] -> [rad]:
  for (auto& val : theta_) val *= M_PI/180.;
  for (auto& val : phi_) val *= M_PI/180.;
  
  createVignettingVector(table);
  
  checkDimensions(table);
}

void NewVignetting::createVignettingVector (CCfits::ExtHDU& table) {
  std::vector<double> vignetting_vector = get_columns (table, "VIGNET", 1);
  
  typedef std::vector<double> d1;
  typedef std::vector<d1> d2;
  typedef std::vector<d2> d3;
  vignetting_array_ = d3(nenergies_, d2(ntheta_, d1(nphi_)));
  
  size_t count=0;
  for(size_t ecount=0; ecount<nenergies_; ecount++){
    for(size_t tcount=0; tcount<ntheta_; tcount++){
      for(size_t pcount=0; pcount<nphi_; pcount++){
        count = ecount + tcount * nenergies_ + pcount * nenergies_ * ntheta_;
        vignetting_array_[ecount][tcount][pcount] = vignetting_vector[count];
      }
    }
  }
}

void NewVignetting::checkDimensions (CCfits::ExtHDU& table) const {
  int vig_num = table.column("VIGNET").index();
  std::string keyName = "TDIM" + std::to_string(vig_num);
  std::string dimensions;
  table.readKey(keyName, dimensions);
  
  removeSubstring(dimensions, "(");
  removeSubstring(dimensions, ")");
  
  std::vector<size_t> vignetting_lengths = separate_string_to_size_t (dimensions, ',');
  if (vignetting_lengths[0]!=nenergies_ ||
    vignetting_lengths[1]!=ntheta_ ||
    vignetting_lengths[2]!=nphi_) {
    throw SixteException("seperations in vignetting file do not correspont to seperate columns!");
  }
}


double NewVignetting::interpolateVigThetaPhi(double psf_theta,
                                             double psf_phi,
                                             const std::vector<std::vector<double>>& vign_array) {
  if (ntheta_ == 0) {
    throw SixteException("Vignetting THETA grid is empty");
  }
  if (nphi_ == 0) {
    throw SixteException("Vignetting PHI grid is empty");
  }

  // --- interpolation in theta (off-axis angle) ---
  size_t theta_low_index = 0;
  size_t theta_high_index = 0;
  double theta_t = 0.0;

  if (ntheta_ == 1) {
    theta_low_index = theta_high_index = 0;
  } else if (psf_theta >= theta_.back()) {
    theta_low_index = theta_high_index = ntheta_ - 1;
  } else {
    // for psf_theta < theta_[0], lower_bound returns begin()+1 -> extrapolation using [0,1]
    auto lower_theta = std::lower_bound(theta_.begin() + 1, theta_.end(), psf_theta);
    long theta_ind = std::distance(theta_.begin(), lower_theta);
    theta_low_index = static_cast<size_t>(theta_ind - 1);
    theta_high_index = static_cast<size_t>(theta_ind);

    double theta_low = theta_[theta_low_index];
    double theta_high = theta_[theta_high_index];
    theta_t = (psf_theta - theta_low) / (theta_high - theta_low);
  }

  // --- interpolation in phi (azimuthal angle) ---
  size_t phi_low_index = 0;
  size_t phi_high_index = 0;
  double phi_t = 0.0;

  if (nphi_ == 1) {
    // No azimuthal sampling.
    phi_low_index = phi_high_index = 0;
  } else {
    const double period = 2.0 * M_PI;

    // Shift psf_phi into the same 2PI-slice as the table, anchored at phi_.front().
    const double phi0 = phi_.front();
    double phi = std::fmod(psf_phi - phi0, period);
    if (phi < 0.0) {
      phi += period;
    }
    phi += phi0;  // now in [phi0, phi0 + 2PI)

    if (phi <= phi_.back()) {
      auto lower_phi = std::lower_bound(phi_.begin() + 1, phi_.end(), phi);
      long phi_ind = std::distance(phi_.begin(), lower_phi);
      phi_low_index = static_cast<size_t>(phi_ind - 1);
      phi_high_index = static_cast<size_t>(phi_ind);

      double phi_low = phi_[phi_low_index];
      double phi_high = phi_[phi_high_index];
      phi_t = (phi - phi_low) / (phi_high - phi_low);
    } else {
      // Wrap interval: [phi_.back(), phi_.front() + 2PI]
      phi_low_index = nphi_ - 1;
      phi_high_index = 0;

      double phi_low = phi_.back();
      double phi_high = phi_.front() + period;
      phi_t = (phi - phi_low) / (phi_high - phi_low);
    }
  }

  // --- bilinear interpolation over theta and phi ---
  const double v_theta0_phi0 = vign_array[theta_low_index][phi_low_index];
  const double v_theta1_phi0 = vign_array[theta_high_index][phi_low_index];
  const double v_interp_phi0 = sixteLerp(v_theta0_phi0, v_theta1_phi0, theta_t);

  const double v_theta0_phi1 = vign_array[theta_low_index][phi_high_index];
  const double v_theta1_phi1 = vign_array[theta_high_index][phi_high_index];
  const double v_interp_phi1 = sixteLerp(v_theta0_phi1, v_theta1_phi1, theta_t);

  return sixteLerp(v_interp_phi0, v_interp_phi1, phi_t);
}


double NewVignetting::getVignettingFactor(double psf_energy, double psf_theta, double psf_phi) {
  if (!is_vignetting_) return 1.;

  /* if we are above or below the defined energies, we return the vignetting
   * value at the edge of the grid  */
  if (psf_energy <= Emin_) {
    return interpolateVigThetaPhi(psf_theta, psf_phi, vignetting_array_[0]);
  }
  if (psf_energy >= Emax_) {
    return interpolateVigThetaPhi(psf_theta, psf_phi, vignetting_array_[nenergies_ - 1]);
  }

  // Find the right energy bin.
  auto lower = std::lower_bound(energy_.begin(), energy_.end(), psf_energy);

  if (lower == energy_.end()) {
    throw SixteException("Failed to find vignetting index");
  }

  long ind = std::distance(energy_.begin(), lower) - 1;

  double vign_val_low  = interpolateVigThetaPhi(psf_theta, psf_phi, vignetting_array_[ind]);
  double vign_val_high = interpolateVigThetaPhi(psf_theta, psf_phi, vignetting_array_[ind + 1]);

  double ifac = (psf_energy - energy_[ind]) / (energy_[ind + 1] - energy_[ind]);
  return sixteLerp(vign_val_low, vign_val_high, ifac);
}

}
