/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/
#include "NewSourceCatalog.h"

#include "NewArf.h"
#include "SixteException.h"

namespace sixte {

NewSourceCatalog::NewSourceCatalog(const std::string& simput_file, const NewArf& arf) {
  int status = EXIT_SUCCESS;
  
  std::unique_ptr<CCfits::FITS> inFile = sixteOpenFITSFileRead(simput_file, "source catalog");
  
  // Set refernce to the random number generator to be used by the
  // SIMPUT library routines.
  setSimputRndGen(sixt_get_random_number);

  // Use the routines from the SIMPUT library to load the catalog.
  simput_ctlg_ = openSimputCtlg(simput_file.c_str(), READONLY, 0, 0, 0, 0, &status);
  CHECK_STATUS_VOID(status)

  // Set reference to ARF for SIMPUT library.
  setSimputARF(simput_ctlg_, (const_cast<ARF *>(arf.c_arf())));

  // Determine the number of point-like and the number of
  // extended sources.
  size_t nextended = 0;
  size_t npointlike = 0;

  double ra_center_img = 0.0;
  double dec_center_img = 0.0;
  
  for (long ii=0; ii<simput_ctlg_->nentries; ii++) {
    // Get the source.
    SimputSrc* src=getSimputSrc(simput_ctlg_, ii+1, &status);
    CHECK_STATUS_BREAK(status)
    
    // Determine whether the source is extended
    bool is_extended = isSimputSrcExtended(simput_ctlg_, src, 0., 0., &status);
    CHECK_STATUS_BREAK(status)
    
    if (is_extended) nextended++;
    else npointlike++;
  }
  
  CHECK_STATUS_VOID(status)
  
  // Allocate memory for an array of the point-like sources,
  // which will be converted into a KDTree afterwards.
  auto* list=(Source*)malloc(npointlike*sizeof(Source));
  CHECK_NULL_VOID(list, status, "memory allocation for source list failed")

  // allocate memory for the array of point-like sources
  extsources_=(Source*)malloc(nextended*sizeof(Source));
  CHECK_NULL_VOID(extsources_, status, "memory allocation for source list failed")
  
  // Empty template object.
  Source* templatesrc=newSource(&status);
  CHECK_STATUS_VOID(status)
  
  // Loop over all entries in the SIMPUT source catalog.
  size_t cpointlike = 0;
  nextsources_ = 0;
  for (long ii=0; ii<simput_ctlg_->nentries; ii++) {
    // Get the source.
    SimputSrc* src = getSimputSrc(simput_ctlg_, ii+1, &status);
    CHECK_STATUS_BREAK(status)
    
    float extension=getSimputSrcExt(simput_ctlg_, src, &ra_center_img, &dec_center_img, 0., 0., &status);
    CHECK_STATUS_BREAK(status)
    
    if (extension>0.) {
      // This is an extended source.
      nextsources_++;
      // Start with an empty Source object for this entry.
      extsources_[nextsources_-1] = *templatesrc;
      // Set the properties from the SIMPUT catalog (position, extension, and row number in the catalog).
      
      // we need the center pixels here of the image, as this is what the extensions refers to)
      extsources_[nextsources_-1].ra  = ra_center_img;
      extsources_[nextsources_-1].dec = dec_center_img;
      extsources_[nextsources_-1].row = ii+1;
      extsources_[nextsources_-1].extension = extension;
      
    } else {
      // This is a point-like source.
      cpointlike++;
      list[cpointlike-1]     = *templatesrc;
      list[cpointlike-1].ra  = src->ra;
      list[cpointlike-1].dec = src->dec;
      list[cpointlike-1].row = ii+1;
    }
  }
  CHECK_STATUS_VOID(status)
  // END of loop over all entries in the FITS table.
  
  // Build a KDTree from the source list (array of Source objects).
  tree_ = buildKDTree2(list, (long)npointlike, 0, &status);
  CHECK_STATUS_VOID(status)
  
  // In a later development stage this could be directly stored in
  // a memory-mapped space.
  
  // Load spectra into the internal cache used by the SIMPUT library.
  // This works only if all spectra are contained as mission-independent
  // spectra in a single binary-table FITS file HDU, which can be found
  // be tracing the location of the spectrum assigned to the first source
  // in the catalog.
  SimputSrc* src = getSimputSrc(simput_ctlg_, 1, &status);
  CHECK_STATUS_VOID(status)
  
  char specref[MAXFILENAME];
  getSimputSrcSpecRef(simput_ctlg_, src, 0., -1.0, specref, &status);
  CHECK_STATUS_VOID(status)
  
  char *search=strchr(specref, ']');
  if (nullptr==search) {
    SIXT_WARNING("no valid reference to source spectrum");
  } else {
    *(search+1)='\0';
    headas_chat(3, "try to load all spectra ('%s') into cache ...\n", specref);
    loadCacheAllSimputMIdpSpec(simput_ctlg_, specref, &status);
    CHECK_STATUS_VOID(&status)
  }
  
  // Release memory.
  if (templatesrc) free(templatesrc);
  free(list);
  
  checkStatusThrow(status, "Failed to load SIMPUT catalog " + simput_file);
}

NewSourceCatalog::~NewSourceCatalog() {
  int status = EXIT_SUCCESS;
  // Free the KD-Tree.
  if (NULL!=tree_) {
    freeKDTreeElement(&tree_);
  }
  // Free the array of extended sources.
  if (NULL!=extsources_) {
    free(extsources_);
  }
  // Free the SIMPUT source catalog.
  if (NULL!=simput_ctlg_) {
    freeSimputCtlg(&simput_ctlg_, &status);
  }
}

NewSourceCatalog::NewSourceCatalog(NewSourceCatalog&& other)
  : tree_(other.tree_),
  extsources_(other.extsources_), nextsources_(other.nextsources_),
  simput_ctlg_(other.simput_ctlg_)
{
  other.tree_=0;
  other.extsources_=0;
  other.simput_ctlg_=0;
}

NewSourceCatalog& NewSourceCatalog::operator=(NewSourceCatalog&& other) {

  tree_ = other.tree_;

  extsources_ = other.extsources_;

  nextsources_ = other.nextsources_;

  simput_ctlg_ = other.simput_ctlg_;

  other.tree_=0;
  other.extsources_=0;
  other.simput_ctlg_=0;

  return *this;
}



LinkedPhoListElement* NewSourceCatalog::sixteGenFoVXRayPhotons(NewAttitude& attitude,
                                                               double fov,
                                                               double time,
                                                               double t1) const {
  SixteVector pointing = attitude.getTelescopeNz(time);
  SixteVector vector{pointing.x(), pointing.y(), pointing.z()};
  
  double mjdref = attitude.mjdref();

  int status = EXIT_SUCCESS;
  
  assert(nullptr!=simput_ctlg_);
  
  // Minimum cos-value for point sources close to the FOV (in the direct
  // neighborhood).
  double close_mult = 1.5;
  double close_fov_min_align = cos(close_mult*fov/2.);
  
  // Perform a range search over all sources in the KDTree and
  // generate new photons for the sources within the FoV.
  // The kdTree only contains point-like sources.
  LinkedPhoListElement* list = KDTreeRangeSearch(tree_, 0, pointing.c_vector(), close_fov_min_align,
                                                 time, t1, mjdref, simput_ctlg_, &status);
  
  // Loop over all extended sources.
  for (size_t ii=0; ii<nextsources_; ii++) {
    // Check if at least a part of the source lies within the FoV.
    Vector location=unit_vector(extsources_[ii].ra,
                                extsources_[ii].dec);
    double min_align = close_mult*(fov*0.5) + extsources_[ii].extension;
    if (min_align > M_PI){
      min_align -= 2*M_PI;
    }
    
    if (!check_fov(&location, pointing.c_vector(), cos(min_align))) {
      
      // Generate photons for this particular source.
      LinkedPhoListElement* newlist = getXRayPhotons(&(extsources_[ii]), simput_ctlg_, time, t1, mjdref, &status);
      CHECK_STATUS_RET(status, list)
      
      // Merge the new photons into the existing list.
      list=mergeLinkedPhoLists(list, newlist);
    }
  }
  
  checkStatusThrow(status, "Failed to create LinkedPhoListElement");
  
  return(list);
}


//const SourceCatalog* NewSourceCatalog::c_sourceCatalog() const {
//  return source_catalog_;
//}


}
