/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "SixteException.h"
#include "SixteVector.h"
#include <string>

#define SIXTE_OUT(...) std::cout << #__VA_ARGS__ << " = " << __VA_ARGS__ << '\n';
#define SIXTE_OUT_HighPrecision(...) std::cout << std::setprecision(15) << #__VA_ARGS__ << " = " << __VA_ARGS__ << '\n';

#define MAXMSG (512)

namespace sixte {

std::pair<std::string, std::string> sixtGetDateTime(double mjdref, double t);

bool isInFov(const SixteVector& x, const SixteVector& x0, double min_align);

// Check near-equality with both absolute and relative tolerances.
// Returns true only if |ref-cur| <= abs_tol AND relative error <= rel_tol.
bool nearEqual(double ref, double cur, double abs_tol, double rel_tol);

void verifyMJDREF(double refmjdref, double mjdref, const std::string& description);

/* --------------- STRING OPERATIONS --------------- */
std::vector<std::string> separate_string_to_strings(const std::string& s_to_s, char dlm);
std::vector<size_t> separate_string_to_size_t(const std::string& s_to_s, char dlm);
std::vector<double> separate_string_to_double(const std::string& s_to_s, char dlm);
std::vector<std::string> separate_fits_string(const std::string& s_to_s, char dlm);

void removeSubstring (std::string& my_string, const std::string& remove);

double sixteLerp (double start, double end, double t);

std::pair<size_t, size_t> getEventsBuffer (size_t amount);

/// For a given value val, returns the index k in a sorted vector vec where vec[k] <= val < vec[k+1],
/// or -1 if no such index exists.
///
/// @tparam T     The type of elements in the vector
/// @param vec    The sorted vector in which to search
/// @param val    The value to be located
/// @return       The index k where vec[k] <= val < vec[k+1],
///               or -1 if no such index exists.
template <typename T>
long findIndexBinarySearch(const std::vector<T>& vec, const T& val) {
  if (val < vec.front() || val >= vec.back()) return -1;

  auto it = std::lower_bound(vec.begin(), vec.end(), val);
  auto index = std::distance(vec.begin(), it);

  if (it == vec.end()) throw SixteException("Invalid binary search index");

  if (it == vec.begin() || val == vec[index]) return index;

  return index - 1;
}

}
