/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "NewSIXT.h"

#include "headas_utils.h"
#include <chrono>
#include <cmath>
#include <iostream>
#include <sstream>

namespace sixte {

std::pair<std::string, std::string> sixtGetDateTime(double mjdref, double t) {
  std::string datestr;
  std::string timestr;
  
  int int_day = (int) (mjdref - 40587.0 + t / 86400.0);
  int int_sec = (int) ((mjdref - 40587.0 - int_day) * 86400.0 + t);
  
  std::tm time_utc = {};
  
  time_utc.tm_sec = int_sec;
  time_utc.tm_min = 0;
  time_utc.tm_hour = 0;
  time_utc.tm_mday = 1 + int_day;
  time_utc.tm_mon = 0;
  time_utc.tm_year = 70;
  time_utc.tm_isdst = -1;
  
  std::time_t timet = std::mktime(&time_utc);
  auto time_utcn = localtime(&timet);
  
  std::strftime(datestr.data(), CHAR_MAX, "%Y-%m-%d", time_utcn);
  std::strftime(timestr.data(), CHAR_MAX, "%H:%M:%S", time_utcn);
  
  return (std::make_pair(datestr.data(), timestr.data()));
}

bool isInFov(const SixteVector& photon_direction, const SixteVector& telescope_nz, double fov_min_align) {
  if (scalarProduct(photon_direction, telescope_nz) < fov_min_align) return false;
  return true;
}

bool nearEqual(double ref, double cur, double abs_tol, double rel_tol) {
  const double abs_diff = std::abs(ref - cur);
  if (abs_diff > abs_tol) return false;
  const double denom = std::abs(ref);
  const double rel = denom > 0 ? abs_diff / denom : abs_diff;  // fall back to abs if ref ~ 0
  return rel <= rel_tol;
}

template<typename T_out>
T_out convert(const std::string& t_in);

/** splitting a comma seperated list of strings to strings, ints or doubles **/
template <>
std::string convert<std::string>(const std::string& t_in) {return t_in;}
template <>
size_t convert<size_t>(const std::string& t_in) {return std::stoul(t_in);}
template <>
double convert<double>(const std::string& t_in) {return std::stod(t_in);}

template <typename T>
std::vector<T> seperate_string (const std::string& s_to_s, const char dlm){
  std::vector<T> vec;
  std::stringstream ss(s_to_s);
  while (ss.good()) {
    std::string str;
    getline(ss, str, dlm);
    T val = convert<T>(str);
    vec.push_back(val);
  }
  return vec;
}

std::vector<std::string> separate_string_to_strings(const std::string& s_to_s, char dlm) {
  return seperate_string<std::string>(s_to_s, dlm);
}
std::vector<size_t> separate_string_to_size_t(const std::string& s_to_s, char dlm) {
  return seperate_string<size_t>(s_to_s, dlm);
}
std::vector<double> separate_string_to_double(const std::string& s_to_s, char dlm) {
  return seperate_string<double>(s_to_s, dlm);
}
std::vector<std::string> separate_fits_string(const std::string &s_to_s, char dlm) {
  // main difference of this function is that separators are ignored
  // if they are within brackets
  std::vector<unsigned int> separator_positions{};
  int depth=0;

  // note down where the separators are
  for (unsigned int i_pos=0; i_pos<s_to_s.size(); i_pos++) {
    const char& c = s_to_s[i_pos];

    if ((c == '[') || (c == '(') || (c == '{')) {depth++;}
    else if ((c == ']') || (c == ')') || (c == '}')) {depth--;}
    else if ((depth == 0) && c == dlm) {
      separator_positions.push_back(i_pos);
    }
  }

  // create individual substrings
  separator_positions.push_back(s_to_s.size());
  std::vector<std::string> substrings;
  unsigned int last=0;
  for (auto s: separator_positions) {
    substrings.push_back(s_to_s.substr(last, s-last));
    last = s+1;
  }

  return substrings;
}

// This function is retired due to updated filename handling,
// but left here in case it might be needed in the future for this or a similar purpuse
//
//std::string changeNameToClobberName(const std::string& filename) {
//  std::vector<std::string> tmp_path = separate_string_to_strings(filename, '/');
//  tmp_path[tmp_path.size()-1] = "!" + tmp_path[tmp_path.size()-1];
//  std::string new_filename;
//  for (const std::string& name:tmp_path) new_filename += (name + "/");
//  new_filename.erase(new_filename.size()-1);
//  return new_filename;
//}

void removeSubstring (std::string& my_string, const std::string& remove) {
  size_t found = my_string.find(remove);
  size_t size_sub = remove.size();
  if (found != my_string.size()) my_string.erase(found, size_sub);
}

double sixteLerp (double start, double end, double t) {
  return (start + t * (end - start));
}

std::pair<size_t, size_t> getEventsBuffer (size_t amount) {
  size_t buffer_amount = 10000;
  size_t buffer_loop = 1;
  
  if (amount > buffer_amount) {
    double double_num_rows = (double)amount / (double)buffer_amount;
    if (amount % buffer_amount == 0) buffer_loop = (long)double_num_rows;
    else buffer_loop = (size_t)(std::ceil(double_num_rows));
  }
  
  return std::make_pair(buffer_loop, buffer_amount);
}

}
