/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <string>
#include <memory>
#include "Ebounds.h"
#include "NewSIXT.h"
#include "Registry.h"

#include "rmf.h"

namespace sixte {

struct RMF* loadRMFWrapper(const std::string& filename);
struct RMF* getRMFWrapper();

struct RMFDeleter {
  void operator()(struct RMF* rmfptr);
};

using RMFUniquePtr = std::unique_ptr<struct RMF, RMFDeleter>;

class NewRMF {
 public:
  explicit NewRMF(const std::string& filename);
  explicit NewRMF(const Ebounds& eb);
  NewRMF();

  void sixteLoadEbounds(const std::string& filename);
  
  [[nodiscard]] long sixteGetEBOUNDSChannel(double energy) const;
  
  [[nodiscard]] std::pair<double, double> sixteGetEBOUNDSEnergyLoHi(long channel) const;

  [[nodiscard]] double sixteGetEBOUNDSEnergy(long channel) const;

  [[nodiscard]] long sampleChannel(double energy) const;
  
  [[nodiscard]] long firstChannel() const;
  
  [[nodiscard]] size_t numberChannels() const;

  [[nodiscard]] long numberEnergyBins() const;

  [[nodiscard]] const struct RMF* rawPtr() const;

  [[nodiscard]] Ebounds exportEbounds() const;

 private:
  std::vector<double> energy_grid_high_;
  mutable std::vector<double> cumulative_response_buffer_;
  mutable std::vector<long> used_channels_;
  bool has_matrix_{false};
  std::string source_filename_;

  RMFUniquePtr rmf_;
};

/**
 * @brief Loader functor for NewRMF objects.
 *
 * Creates a std::shared_ptr<const NewRMF> from an absolute file path.
 */
struct NewRmfLoader {
  /**
   * @brief Construct a NewRMF for the given absolute path.
   *
   * @param abs_path Absolute path to the RMF file.
   * @return Shared pointer to the created NewRMF.
   */
  std::shared_ptr<const NewRMF> operator()(const std::string& abs_path) const {
    return std::make_shared<NewRMF>(abs_path);
  }
};

/**
 * @brief Registry of RMFs keyed by their absolute path.
 */
using RmfRegistry =
    Registry<std::string, std::shared_ptr<const NewRMF>, NewRmfLoader>;

/**
 * @brief Convenience helper: read only the EBOUNDS extension from an RMF file.
 */
std::shared_ptr<Ebounds> loadEboundsOnly(const std::string& abs_path);

}  // namespace sixte
