/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/
#include "NewPhotonfile.h"

#include "SixteException.h"
#include <string>

namespace sixte {

PhotonFile::PhotonFile(const std::string& photon_filename, bool clobber, const ObsInfo& obs_info) {
  auto inFile = sixteOpenFITSFileWrite(photon_filename, clobber, true, "photon file");
  
  auto& table = inFile->pHDU();
  obs_info.addObsInfoKeysToFile(table);
  
  auto& table2 = inFile->extension(1);
  obs_info.addObsInfoKeysToFile(table2);
  
  obs_info.pointing.addPointingInfo(table2);
  
  inFile->destroy();

  fitsfile_ = Fitsfile(photon_filename, FileMode::write);
  fitsfile_.moveToExt("PHOTONS");
}

void PhotonFile::addPhoton2File(const sixte::SixtePhoton &photon) {
  const auto row_num = fitsfile_.getNumRows() + 1;
  
  // Convert from [rad] -> [deg]:
  double phora = 0.;
  double phodec = 0.;
  if (photon.ra()) phora = *photon.ra() * 180. / M_PI;
  else throw SixteException("Failed reading RA of photon!");
  if (photon.dec()) phodec = *photon.dec() * 180. / M_PI;
  else throw SixteException("Failed reading DEC of photon!");
  
  double photime = photon.time();
  double phoenergy = photon.energy();
  long phoph = photon.photon_metainfo().ph_id_;
  long phosrc = photon.photon_metainfo().src_id_;

  fitsfile_.writeCol("TIME", row_num, photime);
  fitsfile_.writeCol("ENERGY", row_num, phoenergy);
  fitsfile_.writeCol("RA", row_num, phora);
  fitsfile_.writeCol("DEC", row_num, phodec);
  fitsfile_.writeCol("PH_ID", row_num, phoph);
  fitsfile_.writeCol("SRC_ID", row_num, phosrc);
}

}
