/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "NewPha2PiLib.h"

#include "sixte_random.h"

#include "healog.h"
#include <utility>

namespace sixte {

NewPha2Pi::NewPha2Pi(std::string filename,
                     const std::string& pirmf_filename,
                     const std::string& specarf_filename)
                     : pha2pi_filename_(std::move(filename)){
  if (pha2pi_filename_.empty()) healog(5) << "No Pha2Pi correction file given ... skipping correction!" << '\n';
  
  std::unique_ptr<CCfits::FITS> inFile = sixteOpenFITSFileRead(pha2pi_filename_, "Pha2Pi file");
  
  // Read the RESPFILE keyword.
  auto &table = inFile->extension("PHA2PI");
  table.readKey("RESPFILE", rmffilecontent_);
  
  if (!pirmf_filename.empty()) pirmf_filename_ = pirmf_filename;
  if (!specarf_filename.empty()) specarf_filename_ = specarf_filename;
  
  nrows_ = table.rows();
  
  try {
    table.column("PHA").read(pha_, 1, nrows_, &nulval_long_);
  } catch (const CCfits::Table::NoSuchColumn &e) {
    table.column("pha").read(pha_, 1, nrows_, &nulval_long_);
  }
  
  // CHECK CONSISTENCY: PHA has to start at zero and must be continuous (assuming sorted PHAs)
  if (pha_[0] != 0 || pha_[nrows_ - 1] != nrows_ - 1) {
    throw SixteException("Pha2Pi correction File is inconsistent! PHA values must be continues starting with 0!");
  }
  
  ngrades_ = table.column("pien").repeat();
  
  for (int ii = 0; ii < nrows_; ii++) {
    std::valarray<double> pien, pilow, pihigh;
    table.column("pien").read(pien, ii+1, &nulval_double_);
    table.column("pilow").read(pilow, ii+1, &nulval_double_);
    table.column("pihigh").read(pihigh, ii+1, &nulval_double_);
    pien_.push_back(pien);
    pilow_.push_back(pilow);
    pihigh_.push_back(pihigh);
  }
}

NewPha2Pi::NewPha2Pi(GenInst* const inst) {
  // Initialize & load Pha2Pi File (NULL if not set)
  std::string pha2pi_filename;
  if (inst->det->pha2pi_filename == nullptr) {
    pha2pi_filename = '\0';
  } else {
    pha2pi_filename = inst->filepath;
    pha2pi_filename += inst->det->pha2pi_filename;
  }
  
  std::string pirmf_filename;
  if (inst->det->pirmf_filename == nullptr) {
    pirmf_filename = '\0';
  } else {
    pirmf_filename = inst->det->pirmf_filename;
  }
  
  std::string specarf_filename;
  if (inst->det->specarf_filename == nullptr) {
    specarf_filename = '\0';
  } else {
    specarf_filename = inst->det->specarf_filename;
  }
  
  // Pha2Pi correction file
  NewPha2Pi p2p(pha2pi_filename, pirmf_filename, specarf_filename);
}

void NewPha2Pi::pha2piCorrectEvent(NewEvent& event, const NewRMF& rmf) {
  if (event.type == -1U || event.pha < 0 || event.pha > pha_[nrows_ - 1]) {   // Do nothing if event is invalid => pi = -1.
    return;
  } else if (event.type < 0 || event.type > long(ngrades_)) {     // Make sure the requested event->type is tabulated
    throw SixteException("Pha2Pi correction event type '" + std::to_string(event.type) + "' not tabulated!" + '\n');
  } else {    // Determine a PI value for the event's PHA value
    if (event.pha != pha_[event.pha]) {
      throw SixteException("pha2pi: Event PHA (" + std::to_string(event.pha) +
                           ") not equal to PHA (" + std::to_string(pha_[event.pha]) +
                           ") in row [" + std::to_string(event.pha) +
                           "] in Pha2Pi file '" + pha2pi_filename_ +
                           "' ... Aborting!" + '\n');
    }

    // Find energy range [emin,emax] corresponding to event's pha
    const double emin = pilow_[event.pha][event.type];
    const double emax = pihigh_[event.pha][event.type];

    if (std::isnan(emin) || std::isnan(emax)) {
      throw SixteException("pha2pi: Pha2Pi file '" + pha2pi_filename_ +
                           "' contains invalid NULL entries for PHA=" + std::to_string(event.pha) +
                           " and TYPE=" + std::to_string(event.type) +
                           "! Aborting ..." + '\n');
    }
    // PI value in keV randomly picked within
    double ran = getUniformRandomNumber();
    const double pi = emin + ran * (emax - emin);

    // PI value in ADU based on given RMF's EBOUNDS
    event.pi = rmf.sixteGetEBOUNDSChannel(pi);
  }
}

void NewPha2Pi::pha2piCorrectEventfile(NewEventfile& evtfile, const std::string& RSPPath, const std::string& RESPfile) {
  healog(3) << "run pha2pi correction on event file ... " << '\n';
  
  evtfile.moveCFITSIOExt("EVENTS");
  std::string eventrmf = evtfile.getKeyStringCFITSIO("RESPFILE");
  
  // CHECK if eventfile & Pha2Pi file were created with the same RMF
  if (!boost::iequals(eventrmf, rmffilecontent_)) {
    throw SixteException("RESPfile keyword of EventFile and Pha2Pi are different, but must be the same!");
  }
  
  // CHECK whether the user demands a different RMF:
  std::string respfile;
  if (!RESPfile.empty()) respfile = RESPfile;
  else respfile = rmffilecontent_;
  
  // We put the paths to the rmf into resppathname:
  std::string resppathname;
  if (!RSPPath.empty()) resppathname = RSPPath + "/" + respfile;
  else resppathname = respfile;
  
  if (!boost::iequals(respfile, rmffilecontent_)) {
    printWarning("pha2pi: Using RMF='" + respfile + "' for PI binning instead of '" + rmffilecontent_ + "'");
  }
  
  // Load the EBOUNDS of the RMF that will be used in the pi correction.
  // Some fields, e.g., FirstChannel, will not be loaded!
  NewRMF rmf;
  rmf.sixteLoadEbounds(resppathname);
  
  // Add 'PI' column to evtfile if necessary
  size_t phi_colnum = evtfile.getColNumCFITSIO("PI", "EVENTS");
  if (phi_colnum == 0) {
    phi_colnum = evtfile.getColNumCFITSIO("PHA", "EVENTS") + 1;
    
    evtfile.addCol2CFITSIOEventFile(phi_colnum, "PI", "J", "ADU");
  }

  // Add pha2pi specific keywords
  evtfile.updateKeyStringCFITSIO("PHA2PI", pha2pi_filename_, "Pha2Pi correction file");
  if (!pirmf_filename_.empty()) {
    evtfile.updateKeyStringCFITSIO("PIRMF", pirmf_filename_, "PI-RMF needed for PI values");
  } else {
    healog(5) << "'PIRMF' Key not written to eventfile as not given!\n";
  }
  if (!specarf_filename_.empty()) {
    evtfile.updateKeyStringCFITSIO("SPECARF", specarf_filename_, "calibrated ARF for analysis");
  } else {
    healog(5) << " 'SPECARF' Key not written to eventfile as not given!\n";
  }

  size_t rows = evtfile.getRowNumCFITSIO("EVENTS");
  
  for (size_t row=1; row <= rows; row++) {
    auto event = evtfile.getEventFromCFITSIOFile(row);
    pha2piCorrectEvent(event, rmf);

    evtfile.updateEvent2CFITSIOFile(event, row);
  }
  
  healog(5) << "... Pha2PI correction successful!\n";
}

}
