/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <string>
#include <valarray>
#include "NewVignetting.h"
#include "NewSIXT.h"
#include "SixtePhoton.h"
#include "Detector.h"
#include "StochInterp.h"

namespace sixte {

/** Stores the PSF data for one particular off-axis angle and one
  particular energy. */
struct PSF_Item_ {
  /** Pointer to the PSF data array [x][y]. */
  std::vector<std::vector<double>> data_array_;
  
  /** Width of the image [pixel]. */
  long naxis1_, naxis2_;
  /** Width of one pixel [m]. */
  double cdelt1_, cdelt2_;
  double crpix1_, crpix2_; /**< [pixel] */
  /** Coordinate value of reference pixel [m]. */
  double crval1_, crval2_;
};

class NewPSF {
 public:
  /** Constructor for the PSF data structure. Reads PSF data from a FITS
    file with one or several image extensions. The file format is
    given by OGIP Calibration Memo CAL/GEN/92-027. The focal length of
    the telescope must be given. If the PSF image in the FITS file is
    defined with a pixel size in [arcsec], this value has to be
    converted to [m] using the knowledge about the focal length. */
  NewPSF(const std::string &filename, double focal_length, const std::string& vig_filename);
  
  /** Calculates the position on the detector, where a photon at given
    sky position with specified energy hits the detector according to
    the PSF data. The exact position is determined with a random
    number generator (randomization over one PSF pixel). The function
    return value is '1', if the photon hits the detector. If it does
    not fall onto the detector, the function returns '0'. The output
    detector position is stored in [m] in the first 2 parameters of
    the function. Output: coordinates of the photon on the detector ([m]). */
  std::optional<Point_2> get_NewPSF_pos(
    /** Incident photon. */
    const SixtePhoton& photon,
    /** Telescope information (pointing directions). */
    const Telescope_attitude& telescope,
    /** Focal length of the telescope [m]. Used for
    off-axis angle calculations. */
    double focal_length);
  
  /** Save the data contained in the PSF data struct to one or several
    image extensions in a FITS file following the OGIP standards for
    2-dimensional PSFs. */
  void saveNewPSFImage(const std::string& filename);
  
  std::vector<double> psf_energy() {
    return energies_;
  };
  std::vector<double> psf_theta() {
    return thetas_;
  };
  std::vector<double> psf_phi() {
    return phis_;
  };
 
 private:
  void addPSFTo3DArray (CCfits::FITS& file, CCfits::HDU& table, double focal_length);
  
  void getValuesForEnergyThetaPhi(CCfits::HDU& table);
  
  void printAvailablePSFEntries ();
  void printAvailablePSFImages (double sum, size_t energy_ind, size_t theta_ind, size_t phi_ind);
  
  /** Different energies PSF images are available for ([keV]). */
  std::vector<double> energies_;
  /** Different off-axis angles PSF images are available for ([rad]). */
  std::vector<double> thetas_;
  /** Different azimuthal angles PSF images are available for ([rad]). */
  std::vector<double> phis_;
  
  /** Interpolator for PSF_Items for different photon energies, off-axis
     angles, and azimuthal angles. */
  StochInterp<3, PSF_Item_> interpolator_{{0,0,0}};
  
  double energy_{};
  double theta_{};
  double phi_{};
  
  NewVignetting vignetting_;
};

/** Add a double value to a list. Before adding the value, check
  whether it is already in the list. In that case it doesn't have to
  be added. The number of list entries is modified appropriately. */
void addDValue2Vector(double value, std::vector<double>& list);

void convertUnits(double& energy_eV, double& theta_arcmin, double& phi_deg);

double createCumulativeDistribution (PSF_Item_& current_psf_item, const std::valarray<double>& dat_arr);

double calculateOffAxisAngleTheta(const SixteVector& telescope_nz, const SixteVector& photon_direction);

double calculateAzimuthalAnglePhi(const SixteVector& telescope_nx, const SixteVector& telescope_ny, const SixteVector& photon_direction);

/** Determine, which PSF should be used for that particular source
    direction and photon energy && Perform a linear interpolation between the next fitting PSF images.
    (Randomly choose one of the neighboring data sets.) */
size_t which_psf(double val, std::vector<double> val_vec);

std::optional<Point_2> samplePosFromPSFImage (const PSF_Item_& psf_item, double focal_length, double theta, double phi_photon, double phi_psf);

void writeHeaderKeywords (CCfits::ExtHDU& image_extension, const PSF_Item_& psf_item, double energy, double theta, double phi, long nhdus);

size_t find_index(double val, std::vector<double> val_vec);

void getWCSKeywords (CCfits::HDU& table, double focal_length, PSF_Item_& current_psf_item);

std::valarray<double> readPSFImage (CCfits::FITS& file, int extension_num);

}
