/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "NewEvent.h"
#include "NewRMF.h"
#include "NewSIXT.h"
#include "ObsInfo.h"
#include "Signal.h"
#include "SixteCCFits.h"
#include "SixteTesEventFile.h"
#include "XMLData.h"
#include <string>
#include <utility>

namespace sixte {

/** Maximum number of photons that are stored as a contribution to a
    single event. If an event originates from more than this
    particular number of photons, the additional ones are not stored
    in the event history. */
#define NEVENTPHOTONS (2)


class NewEventfile {
 public:
  explicit NewEventfile(const std::string& event_filename);
  NewEventfile(const std::string& filename, bool clobber, XMLData& xml_data, const ObsInfo& obs_info,
               std::optional<size_t> rmf_first_channel = std::nullopt, std::optional<size_t> rmf_number_channels = std::nullopt);

  void moveCFITSIOExt(const std::string& ext_name);
  
  size_t getColNumCFITSIO(const std::string& col_name, const std::string& ext_name);
  
  void addEvent2CFITSIOFile(const NewEvent& event);
  
  void updateEvent2CFITSIOFile(const NewEvent& event, size_t nrows);
  
  NewEvent getEventFromCFITSIOFile(size_t row);
  
  size_t getRowNumCFITSIO(const std::string& ext_name);
  
  [[nodiscard]] std::string getKeyStringCFITSIO(const std::string& keyname);
  
  void updateKeyStringCFITSIO(const std::string& keyname, const std::string& keyvar, const std::string& comment);

  template<typename T>
  void updateKeyCFITSIO(const std::string& keyname, const T& value, const std::string& comment);
  
  void addCol2CFITSIOEventFile(size_t columnNumber,
                               const std::string& ttype,
                               const std::string& tform,
                               const std::string& tunit);
  
  // TODO: THIS SHOULD BE CONST CONST!!!
  Fitsfile& fitsfile() {
    return fitsfile_;
  }
  
//  template<typename T>
//  void addCol2EventFile(const std::string& columnName,
//                        CCfits::ValueType type,
//                        long repeatWidth,
//                        T hdu_num_or_name,
//                        const std::string& colUnit=std::string(""),
//                        long decimals = -1,
//                        size_t columnNumber=0) const ;
//
//  template<typename T>
//  void addEvent2File(const NewEvent& event, T hdu_num_or_name);
//
//  template<typename T>
//  void addEvents2File(const std::vector<NewEvent>& events, T hdu_num_or_name);
//
//  template<typename T>
//  void updateEventInFile(size_t row, const NewEvent& event, T hdu_num_or_name);
//
//  template<typename T>
//  void updateEventsInFile(size_t row, const std::vector<NewEvent>& event, T hdu_num_or_name);
//
//  template<typename T>
//  NewEvent getEventFromFile(size_t row, T hdu_num_or_name);
//
//  template<typename T>
//  void updateKeyLong(const std::string& keyname, long keyvar, const std::string& comment, T hdu_num_or_name) const;
//
//  template<typename T>
//  void flushEventBuffer(T hdu_num_or_name);
//
//  template<typename T>
//  std::vector<NewEvent> getEventsFromFile(size_t row, size_t numrows, T hdu_num_or_name);
//
//  template<typename T>
//  [[nodiscard]] size_t getRowNum(T hdu_num_or_name);
//
//  template<typename T>
//  [[nodiscard]] std::string getKeyString(const std::string& keyname, T hdu_num_or_name) const;
//
//  template<typename T>
//  [[nodiscard]] int getColNum(const std::string& colname, T hdu_num_or_name) const;
//
//  template<typename T>
//  void updateKeyString(const std::string& keyname, const std::string& keyvar, const std::string& comment, T hdu_num_or_name) const;
  
 private:
  Fitsfile fitsfile_;
  std::vector<NewEvent> event_buffer_;

//  std::unique_ptr<CCfits::FITS> inFile_;
  template<typename T>
  void updateTLkeywords(std::unique_ptr<CCfits::FITS>& inFile, size_t rmf_first_channel, size_t rmf_number_channels, T hdu_num_or_name);
};

template<typename T>
void NewEventfile::updateTLkeywords(std::unique_ptr<CCfits::FITS>& inFile, size_t rmf_first_channel, size_t rmf_number_channels, T hdu_num_or_name) {
  auto &table = inFile->extension(hdu_num_or_name);
  
  int pha_num = table.column("PHA").index();
  std::string keyname = "TLMIN" + std::to_string(pha_num);
  table.addKey(keyname, rmf_first_channel, "");
  
  keyname = "TLMAX" + std::to_string(pha_num);
  size_t val = rmf_first_channel + rmf_number_channels - 1;
  table.addKey(keyname, val, "");
}

template<typename T>
void NewEventfile::updateKeyCFITSIO(const std::string& keyname, const T& value, const std::string& comment) {
  fitsfile_.updateKey(keyname, value, comment);
}

void copyEvents(NewEventfile& src_file, NewEventfile& dst_file);

//template<typename T>
//// ToDo: check if the optional parameters really need to be given
//void NewEventfile::addCol2EventFile(const std::string& columnName,
//                                    CCfits::ValueType type,
//                                    long repeatWidth,
//                                    T hdu_num_or_name,
//                                    const std::string& colUnit,
//                                    long decimals,
//                                    size_t columnNumber) const {
//  // Insert column in File
//  auto& table = inFile_->extension(hdu_num_or_name);
//  if (columnNumber) table.addColumn(type, columnName, repeatWidth, colUnit, decimals, columnNumber);
//  else {
//    table.addColumn(type, columnName, repeatWidth, colUnit, decimals);
//    columnNumber = table.column("columnName").index();
//  }
//  // Set TUNIT of added column
//  std::string keystring = "TUNIT" + std::to_string(columnNumber);
//  std::string comment = "Unit of column " + std::to_string(type);
//
//  std::string t_form = returnTFORMAT(type);
//  table.addKey(keystring, t_form, comment);
//}
//
//template<typename T>
//void NewEventfile::updateEventInFile(size_t row, const NewEvent& event, T hdu_num_or_name) {
//  auto& table = inFile_->extension(hdu_num_or_name);
//
//  updateEvent(event, table, row);
//}
//
//template<typename T>
//void NewEventfile::updateEventsInFile(size_t row, const std::vector<NewEvent>& events, T hdu_num_or_name) {
//  auto &table = inFile_->extension(hdu_num_or_name);
//  updateEvents(events, table, row);
//  for (const auto& event : events) event_buffer_.emplace_back(event);
//  if (event_buffer_.size() == 10000) {
//    auto &table = inFile_->extension(hdu_num_or_name);
//    updateEvents(event_buffer_, table, row);
//    event_buffer_.clear();
//  }
//}
//
//template<typename T>
//void NewEventfile::flushEventBuffer(T hdu_num_or_name) {
//  if (event_buffer_.empty()) return;
//  long row = inFile_->extension(hdu_num_or_name).rows();
//  auto &table = inFile_->extension(hdu_num_or_name);
//  updateEvents(event_buffer_, table, row);
//  event_buffer_.clear();
//}
//
//template<typename T>
//void NewEventfile::addEvent2File(const NewEvent& event, T hdu_num_or_name) {
//  long nrows = inFile_->extension(hdu_num_or_name).rows();
//  if (first_row_) first_row_ = false;
//  else nrows += 1;
//  updateEventInFile(nrows, event, hdu_num_or_name);
//}
//
//template<typename T>
//void NewEventfile::addEvents2File(const std::vector<NewEvent>& events, T hdu_num_or_name) {
//  long nrows = inFile_->extension(hdu_num_or_name).rows();
//  if (first_row_) first_row_ = false;
//  else nrows += 1;
//  updateEventsInFile(nrows, events, hdu_num_or_name);
//}
//
//template<typename T>
//NewEvent NewEventfile::getEventFromFile(size_t row, T hdu_num_or_name) {
//  flushEventBuffer(hdu_num_or_name);
//  auto &table = inFile_->extension(hdu_num_or_name);
//
//  return getEvent(table, row);
//}
//
////TODO: This should be const
//template<typename T>
//std::vector<NewEvent> NewEventfile::getEventsFromFile(size_t row, size_t numrows, T hdu_num_or_name) {
//  flushEventBuffer(hdu_num_or_name);
//  auto &table = inFile_->extension(hdu_num_or_name);
//
//  return getEvents(table, row, numrows);
//}
//
//
//template<typename T>
//size_t NewEventfile::getRowNum(T hdu_num_or_name) {
//  flushEventBuffer(hdu_num_or_name);
//  return inFile_->extension(hdu_num_or_name).rows();
//}
//
//template<typename T>
//std::string NewEventfile::getKeyString(const std::string& keyname, T hdu_num_or_name) const {
//  std::string keyvar;
//  inFile_->extension(hdu_num_or_name).readKey(keyname, keyvar);
//  if (keyvar.empty()) return "";
//  else return keyvar;
//}
//
//template<typename T>
//int NewEventfile::getColNum(const std::string& colname, T hdu_num_or_name) const {
//  int index = inFile_->extension(hdu_num_or_name).column(colname).index();
//  return index;
//}
//
//template<typename T>
//void NewEventfile::updateKeyString(const std::string& keyname, const std::string& keyvar, const std::string& comment, T hdu_num_or_name) const {
//  inFile_->extension(hdu_num_or_name).addKey(keyname, keyvar, comment);
//}
//
//template<typename T>
//void NewEventfile::updateKeyLong(const std::string& keyname, long keyvar, const std::string& comment, T hdu_num_or_name) const{
//  inFile_->extension(hdu_num_or_name).addKey(keyname, keyvar, comment);
//}

// TODO: add for all T hdu_num_or_name functions in c++20:
// requires std::is_unsigned<T>::value || std::is_same<T, const char *>::value ||  std::is_same<T, std::string>::value

}
