/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "NewEventfile.h"

#include "NewRMF.h"
#include "NewSIXT.h"
#include "SixteException.h"

namespace sixte {

NewEventfile::NewEventfile(const std::string& event_filename)
    : fitsfile_(event_filename, FileMode::write) {
  fitsfile_.moveToExt("EVENTS");
}

NewEventfile::NewEventfile(const std::string& event_filename,
                           bool clobber,
                           XMLData& xml_data,
                           const ObsInfo& obs_info,
                           std::optional<size_t> rmf_first_channel,
                           std::optional<size_t> rmf_number_channels){
  if (event_filename.empty()) throw SixteException("Eventfile name not specified!");
  
  // CCFITS routine necessary because of templates
  auto inFile = sixteOpenFITSFileWrite(event_filename, clobber, true, "event file");

  auto& table_pHDU = inFile->pHDU();
  obs_info.addObsInfoKeysToFile(table_pHDU);
  table_pHDU.addKey("HDUCLAS1", "EVENTS", "");
  table_pHDU.addKey("HDUCLASS", "OGIP", "");

  auto& table = inFile->extension("EVENTS");
  obs_info.addObsInfoKeysToFile(table);

  // Update the TLMIN and TLMAX keywords for the DETX and DETY columns.
  
  int rawx_colnum = table.column("RAWX").index();
  int rawy_colnum = table.column("RAWY").index();

  auto detector = xml_data.child("detector");

  // Check if geometry node exists, otherwise use detector directly
  // (for backward compatibility)
  auto parent_node = detector.optionalChild("geometry").value_or(detector);
  
  auto dimensions = parent_node.child("dimensions");
  int x_dim = dimensions.attributeAsInt("xwidth");
  int y_dim = dimensions.attributeAsInt("ywidth");

  table.addKey("TLMIN" + std::to_string(rawx_colnum), 0, "");
  table.addKey("TLMAX" + std::to_string(rawx_colnum), x_dim - 1, "");
  table.addKey("TLMIN" + std::to_string(rawy_colnum), 0, "");
  table.addKey("TLMAX" + std::to_string(rawy_colnum), y_dim - 1, "");

  if (table.column("PH_ID").repeat() != table.column("SRC_ID").repeat() &&
      table.column("PH_ID").repeat() != table.column("TEL_ID").repeat()
      && table.column("PH_ID").repeat() != NEVENTPHOTONS) {
    throw SixteException("inconsistent maximum number of photons contributing to a single event "
                         "(simulation: " + std::to_string(NEVENTPHOTONS)
                           + ", event file template " + std::to_string(table.column("PH_ID").repeat()) + ")");
  }
  if (xml_data.root().attributeAsString("telescop") == "THESEUS"
      && xml_data.root().attributeAsString("instrume") == "SXI" ) {
    table.addColumn(CCfits::Tdouble, "ORIGINVEC", 3);
  }
  
  // Additional information needed by e.g. ero_calevents
  double rota = parent_node.child("wcs").attributeAsDoubleOr("rota", 0.0);
  table.addKey("CCDROTA", rota, "CCD rotation angle [deg]");
  
  if (!rmf_first_channel || !rmf_number_channels) {
    std::string rmf_filename = xml_data.dirname() +
      xml_data.child("detector").child("rmf").attributeAsString("filename");
    NewRMF rmf(rmf_filename);
    rmf_first_channel = rmf.firstChannel();
    rmf_number_channels = rmf.numberChannels();
  }
  
  updateTLkeywords(inFile, *rmf_first_channel, *rmf_number_channels, 1U);
  table.addKey("EXPOSURE", obs_info.total_sim_time, "exposure time [s]");
  obs_info.pointing.addPointingInfo(table);
  
  inFile->destroy();
  fitsfile_ = Fitsfile(event_filename, FileMode::write);
  fitsfile_.moveToExt("EVENTS");
}

std::string NewEventfile::getKeyStringCFITSIO(const std::string& keyname) {
  std::string keyvar;
  fitsfile_.readKey(keyname, keyvar);
  if (keyvar.empty()) return "";
  else return keyvar;
}

NewEvent NewEventfile::getEventFromCFITSIOFile(size_t row) {
  return getEventCFITSIO(fitsfile_, (long)row);
}

void NewEventfile::addEvent2CFITSIOFile(const NewEvent& event) {
  const auto row_num = fitsfile_.getNumRows() + 1;
  updateEvent2CFITSIOFile(event, row_num);
}

void NewEventfile::updateEvent2CFITSIOFile(const NewEvent& event, size_t row) {
  updateEventCFITSIO(event, fitsfile_, row);
}

size_t NewEventfile::getRowNumCFITSIO(const std::string& ext_name) {
  fitsfile_.moveToExt(ext_name);
  return fitsfile_.getNumRows();
}

size_t NewEventfile::getColNumCFITSIO(const std::string& col_name, const std::string& ext_name) {
  fitsfile_.moveToExt(ext_name);
  return fitsfile_.getColNum(col_name);
}

void NewEventfile::moveCFITSIOExt(const std::string& ext_name) {
  fitsfile_.moveToExt(ext_name);
}

void NewEventfile::addCol2CFITSIOEventFile(size_t columnNumber,
                                           const std::string& ttype,
                                           const std::string& tform,
                                           const std::string& tunit) {
  
  if (!columnNumber) columnNumber = fitsfile_.getNumCols();

  fitsfile_.addCol(columnNumber, ttype, tform, tunit);
}

void NewEventfile::updateKeyStringCFITSIO(const std::string& keyname, const std::string& value, const std::string& comment) {
  fitsfile_.updateKey(keyname, value, comment);
}

void copyEvents(NewEventfile& src_file, NewEventfile& dst_file) {
  src_file.moveCFITSIOExt("EVENTS");
  dst_file.moveCFITSIOExt("EVENTS");

  size_t num_events = src_file.getRowNumCFITSIO("EVENTS");

  for (size_t ii = 1; ii <= num_events; ++ii) {
    NewEvent event = src_file.getEventFromCFITSIOFile(ii);
    dst_file.addEvent2CFITSIOFile(event);
  }
}

}
