/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "Ebounds.h"
#include "Signal.h"
#include "SixteCCFits.h"

#include <string>
#include <vector>

namespace sixte {

const size_t NEW_NEVENTPHOTONS = 2;
const size_t SIGNAL_MATRIX_SIZE = 9;

/** Event on a pixelized X-ray detector. */
class NewEvent {
 public:
  NewEvent() = default;
  
  /** former: addSignalToEventFile */
  NewEvent(const Signal &pixel_signal,
           const Ebounds &ebounds,
           size_t xindex,
           size_t readoutindex,
           double readout_time,
           long frame_number);
  
  /** Time of detection [s]. */
  double time{0};
  /** Frame counter. */
  long frame{0};
  /** Detected PHA channel [adu]. */
  long pha{0};
  /** PI value derived from PHA channel [adu]. */
  long pi{0};
  /** Signal in [keV]. */
  double signal{0};
  /** Raw detector coordinates. Indices start at 0. */
  size_t rawx{0}, rawy{0};
  /** Back-projected right ascension to the sky [rad]. */
  double ra{0};
  /** Back-projected declination to the sky [rad]. */
  double dec{0};
  /** Identifiers of the contributing photons. */
  std::vector<long> ph_id;
  /** Identifiers of the corresponding sources (defined in the SIMPUT
      source catalog). */
  std::vector<long> src_id;
  /** optional origin vector for instruments following a spherical geometry.
      Only one per event.  */
  std::optional<SixteVector> origin;
  /** Number of pixels involved in the split pattern. This number is
      one for a single-pixel event. */
  long npixels{0};
  /** Split pattern type: single (0), double (1-4), triple (5-8),
      quadruple (9-12), or invalid (-1). */
  long type{0};
  /** Pile-up flag. */
  size_t pileup{0};
  /** 3x3 array with individual signal values [keV] around the main
      event. */
  std::vector<double> signals;
  /** 3x3 array with individual energy channels [adu] around the main
      event. */
  std::vector<long> phas;
};

NewEvent getEventCFITSIO(Fitsfile& fitsfile, size_t row);
void updateEventCFITSIO(const NewEvent& event, Fitsfile& fitsfile, size_t row);

class MicroCalEvent {
 public:
  MicroCalEvent() = default;
  /** Time of detection [s]. */
  double time{0};
  /** Signal in [keV]. */
  double signal{0};
  
  long grade1{0};
  long grade2{0};
  /** Unique Pixel ID. */
  size_t pixid{0};
  /** Identifiers of the contributing photons. */
  std::vector<long> ph_id;
  /** Back-projected right ascension to the sky [rad]. */
  double ra{0};
  /** Back-projected declination to the sky [rad]. */
  double dec{0};
  
  double detx{0.};
  double dety{0.};
  
  /** Detected PHA channel [adu]. */
  long pha{0};
  
  long grading{0};
  
  /** Identifiers of the corresponding sources (defined in the SIMPUT
      source catalog). */
  std::vector<long> src_id;
  
  long n_xt{0};
  long e_xt{0};
};

} // sixte