/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "NewEvent.h"

namespace sixte {

NewEvent::NewEvent(const Signal &pixel_signal,
                   const Ebounds &ebounds,
                   size_t xindex,
                   size_t readoutindex,
                   double readout_time,
                   long frame_number) {
  signal = pixel_signal.val();
  
  // Copy the information about the original photons
  const auto n_meta = pixel_signal.photon_metainfo().size();
  ph_id.reserve(n_meta);
  src_id.reserve(n_meta);

  for (size_t ii = 0; ii < n_meta; ii++) {
    ph_id.push_back(pixel_signal.photon_metainfo()[ii].ph_id_);
    src_id.push_back(pixel_signal.photon_metainfo()[ii].src_id_);
  }

  if (pixel_signal.photon_metainfo()[0].origin_) {
    origin = pixel_signal.photon_metainfo()[0].origin_;
  }

  // Apply the detector response
  pha = ebounds.sixteGetEBOUNDSChannel(pixel_signal.val());
  
  // Store remaining information.
  rawy = readoutindex;
  rawx = xindex;
  time = readout_time;  // Time of detection.
  frame = frame_number; // Frame of detection.
  npixels = 1;

  signals.assign(SIGNAL_MATRIX_SIZE, 0.0);
  phas.assign(SIGNAL_MATRIX_SIZE, 0L);
}

NewEvent getEventCFITSIO(Fitsfile& fitsfile, size_t row) {
  NewEvent event;

  fitsfile.readCol("TIME", row, event.time);
  fitsfile.readCol("FRAME", row, event.frame);
  fitsfile.readCol("PHA", row, event.pha);
  fitsfile.readCol("SIGNAL", row, event.signal);
  fitsfile.readCol("RAWX", row, event.rawx);
  fitsfile.readCol("RAWY", row, event.rawy);
  fitsfile.readCol("RA", row, event.ra);
  event.ra *= M_PI / 180.;
  fitsfile.readCol("DEC", row, event.dec);
  event.dec *= M_PI / 180.;

  fitsfile.readCol("PH_ID", row, NEW_NEVENTPHOTONS, event.ph_id);
  fitsfile.readCol("SRC_ID", row, NEW_NEVENTPHOTONS, event.src_id);

  fitsfile.readCol("NPIXELS", row, event.npixels);
  fitsfile.readCol("TYPE", row, event.type);
  fitsfile.readCol("PILEUP", row, event.pileup);

  fitsfile.readCol("SIGNALS", row, SIGNAL_MATRIX_SIZE, event.signals);
  fitsfile.readCol("PHAS", row, SIGNAL_MATRIX_SIZE, event.phas);

  fitsfile.readCol("PI", row, event.pi);

  if (fitsfile.checkColExists("ORIGINVEC")) {
    fitsfile.readCol("ORIGINVEC", row, *event.origin);
  }

  return event;
}

void updateEventCFITSIO(const NewEvent& event, Fitsfile& fitsfile, size_t row) {
  fitsfile.writeCol("TIME", row, event.time);
  fitsfile.writeCol("FRAME", row, event.frame);
  fitsfile.writeCol("PHA", row, event.pha);
  fitsfile.writeCol("SIGNAL", row, event.signal);
  fitsfile.writeCol("RAWX", row, event.rawx);
  fitsfile.writeCol("RAWY", row, event.rawy);
  fitsfile.writeCol("RA", row, event.ra * 180. / M_PI);
  fitsfile.writeCol("DEC", row, event.dec * 180. / M_PI);
  
  // Use thread-local buffers to avoid memory allocations
  thread_local std::vector<long> ph_id_buffer(2);
  thread_local std::vector<long> src_id_buffer(2);
  
  if (event.ph_id.size() >= 2) {
    ph_id_buffer[0] = event.ph_id[0];
    ph_id_buffer[1] = event.ph_id[1];
  } else {
    ph_id_buffer[0] = event.ph_id[0];
    ph_id_buffer[1] = 0;
  }
  fitsfile.writeCol("PH_ID", row, ph_id_buffer.size(), ph_id_buffer);
  
  if (event.src_id.size() >= 2) {
    src_id_buffer[0] = event.src_id[0];
    src_id_buffer[1] = event.src_id[1];
  } else {
    src_id_buffer[0] = event.src_id[0];
    src_id_buffer[1] = 0;
  }
  fitsfile.writeCol("SRC_ID", row, src_id_buffer.size(), src_id_buffer);

  fitsfile.writeCol("NPIXELS", row, event.npixels);
  fitsfile.writeCol("PILEUP", row, event.pileup);
  fitsfile.writeCol("TYPE", row, event.type);

  assert(event.signals.size() == SIGNAL_MATRIX_SIZE);
  fitsfile.writeCol("SIGNALS", row, event.signals.size(), event.signals);
  assert(event.phas.size() == SIGNAL_MATRIX_SIZE);
  fitsfile.writeCol("PHAS", row, event.phas.size(), event.phas);
  
  if (fitsfile.checkColExists("ORIGINVEC")) {
    fitsfile.writeCol("ORIGINVEC", row, *event.origin);
  }

  // only write PI value if event->pi value is valid, i.e., != -1.
  if (event.pi != -1){
    fitsfile.writeCol("PI", row, event.pi);
  }
}

}