/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "NewSIXT.h"
#include "ObsInfo.h"
#include "SimulationParameters.h"
#include "SixteStructs.h"
#include "Telescope_attitude.h"
#include "wcs.h"
#include <string>

namespace sixte {

/** Entry of the Attitude collection of pointing directions. */
class NewAttitudeEntry{
 public:
  NewAttitudeEntry(double time_in, double ra_in, double dec_in, double rollangle_in);
  
  /** Point of time for which this attitude is valid. */
  double time;
  
  /** Telescope pointing direction. */
  SixteVector nz{};
  
  /** Roll-angle ([rad]). */
  double roll_angle;
};

class NewAttitude {
 public:
  NewAttitude(const ObsPointing& pointing, const ObsTime& time, double dt);
  
  NewAttitude(const std::string& filename, double dt);
  NewAttitude(double dt,
              double mjdref,
              double tstart,
              double tstop,
              double ra,
              double dec,
              double roll_angle);
  
  NewAttitude(double amplitude,
              double ra0,
              double dec0,
              double tstart,
              double tstop,
              double mjdref,
              double dt,
              size_t npoints);
  
  /** Produce a pointing attitude. */
  void setupPointingAttitude(double mjdref,
                             double tstart,
                             double tstop,
                             double ra,
                             double dec,
                             double roll_angle);
  
  /** Check if the interval specified by tstart and tstop with respect to mjdred
  * is covered by the attitude. If that is not the case, an error is returned. */
  void checkAttitudeTimeCoverage(double mjdref, double tstart, double tstop) const;
  void setWCScurrentPointing(const SixteVector& nz, struct wcsprm* wcs);
  std::pair<double, double> getCurrentVelocity(double att_start, double att_stop);
  
  void setAttitudeCurrEntry(double time);
  
  /** Determine the telescope pointing direction at a specific time. */
  SixteVector getTelescopeNz(double time);
  
  /** Determine the 3 axes vectors for the telescope coordinate system. */
  Telescope_attitude getTelescopeAxes(double time);
  
  /** Determine the roll-angle ([rad]) at a specific time. */
  double getRollAngle(double time);
  
  
  
  /** Write an attitude to a FITS file */
  void write_attitude(const std::string& filename, bool clobber);
  
  [[nodiscard]] double mjdref() const {
    return mjdref_;
  }
  
  [[nodiscard]] double dt() const {
    return dt_;
  }
  
 private:
  void getDataFromAttFile(const std::string& filename);
  
  std::pair<double, double> getRaDec (size_t row_num);
  static std::pair<double, double> diffRaDec (std::pair<double, double> firstRaDec,
                                              std::pair<double, double> secondRaDec);
  
  /** Number of AttituideEntry elements in the Attitude. */
  size_t nentries_{};
  
  /** Individual AttitudeEntry elements giving the attitude of the
      telescope at a particular point of time. */
  std::vector<NewAttitudeEntry> entries_;
  
  /** Index of the currently selected entry in the collection. */
  long currentry_{};
  
  /** Alignment flag. Determines the reference direction for the
      alignment of the nx vector. The nx vector is determined by the
      rotation of this reference direction around the roll angle. */
  AttNxAlign align_{};
  
  double mjdref_{};
  double tstart_{};
  double tstop_{};
  double dt_{};
};

/** calculate a lissajous pattern from time 0 to 1, with Amplitude ampl */
lissajous_pattern_struct lissajous_pattern(double amplitude, size_t nentries);

/**  % following Carroll & Ostlie, Chapter 24.3, Coordinates obtained with HEASARC tools
 *      (taken from the Remeis ISISscripts, initially written by Moritz Böck)
 *
 *      expectiong l=world[0] and b=world[1]
 *      output     RA=world[1] and dec=world[1]
 */
void convert_galLB2RAdec(std::vector<double> world);

NewAttitudeEntry read_attitude_file(long row, Fitsfile& table);

}





















