/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/
#include "NewAttitude.h"

#include "SixteCCFits.h"
#include "SixteException.h"

#include "healog.h"
#include <boost/algorithm/string/predicate.hpp>

namespace sixte {
NewAttitude::NewAttitude(const ObsPointing& pointing, const ObsTime& time, double dt) {
  std::string filename = pointing.attitude_file;
  dt_ = dt;
  if (filename.empty()) {
    setupPointingAttitude(time.mjdref,
                          time.tstart,
                          time.tstart+time.exposure,
                          pointing.ra,
                          pointing.dec,
                          pointing.rollangle);
    return;
  }
  getDataFromAttFile(filename);
}

NewAttitude::NewAttitude(const std::string& filename, double dt)
:dt_(dt) {
  if (filename.empty()) throw SixteException("Wrong Contructor for Attitude!");
  getDataFromAttFile(filename);
}

NewAttitude::NewAttitude(double dt,
                         double mjdref,
                         double tstart,
                         double tstop,
                         double ra,
                         double dec,
                         double roll_angle) : dt_(dt) {
  setupPointingAttitude(mjdref, tstart, tstop, ra, dec, roll_angle);
}

// ampl given in [arcsec]
NewAttitude::NewAttitude(double amplitude, double ra0, double dec0,
                         double tstart, double tstop, double mjdref, double dt, size_t npoints) : dt_(dt) {
  // for this setup 1000 points make a smooth curve
  nentries_ = npoints;
  
  mjdref_ = mjdref;
  tstart_ = tstart;
  tstop_ = tstop;
  
  auto lps = lissajous_pattern(amplitude, nentries_);
  
  std::vector<double> ra = lps.x_val;
  std::vector<double> dec = lps.y_val;
  
  for (size_t ii = 0; ii < nentries_; ii++) {
    entries_.emplace_back( ((tstop - tstart) * lps.time[ii]) + tstart,
                           ra[ii] + ra0,
                           dec[ii] + dec0,
                           0 );
  }
}

void NewAttitude::getDataFromAttFile(const std::string& filename) {
  
  std::unique_ptr<CCfits::FITS> inFile;
  inFile = sixteOpenFITSFileRead(filename, "attitude catalog file");

  auto &table = inFile->extension(1);
  
  // Determine whether the roll angle alignment refers to the telescope's direction of motion or to the equatorial plane.
  std::string alignment;
  try {
    table.readKey("ALIGNMEN", alignment);
  } catch (const CCfits::HDU::NoSuchKeyword&) {
    alignment="";
  }

  // Check the value of the header keyword and set the alignment flag
  // appropriately.
  if (alignment.empty() || boost::iequals(alignment, "NORTH")) {
    align_ = AttNxAlign::ATTNX_NORTH;
  } else if (boost::iequals(alignment, "MOTION")) {
    align_ = AttNxAlign::ATTNX_MOTION;
  } else {
    throw SixteException("invalid value for keyword ALIGNMEN in attitude file");
  }
  
  table.readKey("MJDREF", mjdref_);
  table.readKey("TSTART", tstart_);
  table.readKey("TSTOP", tstop_);
  
  // Check if TIMEZERO is present. If yes, it must be zero.
  std::optional<double> timezero = std::nullopt;
  
  double timezero_temp;
  try {
    table.readKey("TIMEZERO", timezero_temp);
    timezero = timezero_temp;
  } catch (CCfits::ExtHDU::NoSuchKeyword &e) {
    healog(3) << "TIMEZERO keyword not given... continue" << '\n';
  }
  
  if (timezero && *timezero != 0.0) {
    throw SixteException("implementation requires that TIMEZERO=0.0 in input file");
  }
  
  nentries_ = table.rows();
  
  inFile->destroy();
  auto fitsfile = Fitsfile(filename, FileMode::read);
  fitsfile.moveToExt(1);
  
  for (size_t row = 1; row <= nentries_; row++) {
    entries_.emplace_back(read_attitude_file((long)row, fitsfile));
  }
}

void NewAttitude::setAttitudeCurrEntry(double time) {
  // Check if this is a pointing attitude with only one data point.
  if (nentries_ == 1) {
    currentry_ = 0;
    return;
  }
  
  // Check if the requested time lies within the current time bin.
  while (time < entries_[currentry_].time) {
    // Check if the beginning of the Attitude is reached.
    if (currentry_ <= 0) {
      throw SixteException("no attitude entry available for time " + std::to_string(time));
    }
    // If not, go one step back.
    currentry_--;
  }
  
  while (time > entries_[currentry_ + 1].time) {
    // Check if the end of the Attitude is reached.
    if (currentry_ >= (long)nentries_ - 2) {
      throw SixteException("no attitude entry available for time " + std::to_string(time));
    }
    // If not, go one step further.
    currentry_++;
  }
}

SixteVector NewAttitude::getTelescopeNz(double time) {
  if (nentries_ > 1) { //Survey attitude
    setAttitudeCurrEntry(time);
    return interpolateCircleVector(entries_[currentry_].nz,
                                   entries_[currentry_ + 1].nz,
                                   (time - entries_[currentry_].time) /
                                   (entries_[currentry_ + 1].time - entries_[currentry_].time));
  } else { // Pointing attitude
    return entries_[0].nz;
  }
}

Telescope_attitude NewAttitude::getTelescopeAxes(double time) {
  SixteVector nz = getTelescopeNz(time);
  // After calling getTelescopeNz() the internal current entry pointer
  // within the attitude catalog is set to the appropriate time bin.
  
  bool pointed = false;
  SixteVector dnz;
  if (nentries_ == 1) pointed = true;
  else {
    if (currentry_ > 0) dnz = vectorDifference(nz, entries_[currentry_].nz);
    else dnz = vectorDifference(entries_[currentry_ + 1].nz, nz);
    
    if (scalarProduct(dnz, dnz) < 1.e-10) pointed = true;
  }
  
  // Check if the x1 vector should be aligned along the north direction
  // or along the direction of motion of the telescope axis
  // (neglecting the rotation according to the roll angle). For a pointed
  // observation the alignment is done along the north direction.
  SixteVector x1;
  if (pointed || (AttNxAlign::ATTNX_NORTH == align_)) {
    // Check if the telescope is pointing towards one of the poles.
    double x_alignment = fabs(scalarProduct(nz, {1.0, 0.0, 0.0}));
    double y_alignment = fabs(scalarProduct(nz, {0.0, 1.0, 0.0}));
    
    if ((x_alignment < 1.e-20) && (y_alignment < 1.e-20)) {
      x1 = {1.0, 0.0, 0.0};
    } else {
      // If not, align the x1 vector along the north direction.
      x1 = {0.0, 0.0, 1.0};
    }
  } else if (AttNxAlign::ATTNX_MOTION == align_) {
    x1 = normalizeVector(dnz);
  } else {
    throw SixteException("invalid attitude alignment");
  }
  
  // Subtract the projection of the pointing direction such that x1 and nz are perpendicular to each other.
  double scp = scalarProduct(x1, nz);
  x1 -= (nz * scp);
  
  x1 = normalizeVector(x1);
  
  // Determine the y1 vector, which is perpendicular to the other two axes nz and x1:
  SixteVector y1 = normalizeVector(crossProduct(nz, x1));

  double roll_angle = getRollAngle(time);
  SixteVector nx = x1 * cos(roll_angle) + y1 * sin(roll_angle);
  SixteVector ny = (x1 * sin(roll_angle) * -1) + y1 * cos(roll_angle);
  
  return {nx, ny, nz};
}

double NewAttitude::getRollAngle(const double time) {
  if (nentries_ > 1) {
    setAttitudeCurrEntry(time);
    double previous_time_step = entries_[currentry_].time;
    double next_time_step = entries_[currentry_ + 1].time;

    double time_step = (time - previous_time_step) / (next_time_step - previous_time_step);
    double interpolate_roll_angle = (entries_[currentry_].roll_angle * (1. - time_step) +
                                      entries_[currentry_ + 1].roll_angle * time_step);
    
    return interpolate_roll_angle;
    
  } else { // Pointing attitude.
    return (entries_[0].roll_angle);
  }
}

void NewAttitude::setupPointingAttitude(double mjdref,
                                        double tstart,
                                        double tstop,
                                        double ra,
                                        double dec,
                                        double roll_angle) {
  nentries_ = 1;
  mjdref_ = mjdref;
  tstart_ = tstart;
  tstop_ = tstop;

  entries_.emplace_back(tstart, ra, dec, roll_angle);
}

void NewAttitude::checkAttitudeTimeCoverage(const double mjdref, const double tstart, const double tstop) const {
  if (fabs(mjdref_-mjdref)>1.e-6) {
    throw SixteException("MJDREF in attitude file does not match required value of " + std::to_string(mjdref));
  }
  
  if ((tstart_ > tstart) || (tstop_ < tstop)) {
    throw SixteException("attitude does not cover the required period from "
                         + std::to_string(tstart) + " to " + std::to_string(tstop));
  }
}

void NewAttitude::setWCScurrentPointing(const SixteVector& nz, struct wcsprm *wcs) {
  std::pair<double, double> currRaDec = getRaDec(currentry_);
  
  std::pair<double, double> nextRaDec = getRaDec(currentry_+1);
  
  std::pair<double, double> diffRADEC = diffRaDec(currRaDec, nextRaDec);
  
  //get dec from actual current pointing (nz is approximated and lies in between the attitude entries)
  //formula is from unit vector 'backwards'
  double dec_nz = asin(nz.z()) * 180. / M_PI; //could be the wrong value from asin
  
  if (fabs(currRaDec.second - dec_nz) > diffRADEC.second) {    //has to lie inbetween the attitude-interval
    dec_nz = 180. - dec_nz;                 //if not, use the other solution
  }
  
  double ra_nz = asin(nz.y() / cos(dec_nz * M_PI / 180.)) * 180. / M_PI;
  if (fabs(currRaDec.first - ra_nz) > diffRADEC.first) {
    ra_nz = 360. - ra_nz;
  }
  
  wcs->crval[0] = ra_nz;  //in deg
  wcs->crval[1] = dec_nz;
}

std::pair<double, double> NewAttitude::getCurrentVelocity(double att_start, double att_stop) {
  std::pair<double, double> currRaDec = getRaDec(currentry_);
  std::pair<double, double> nextRaDec = getRaDec(currentry_+1);
  std::pair<double, double> diffRADEC = diffRaDec(currRaDec, nextRaDec);
  
  double interval = att_stop - att_start;
  
  double vel_ra = diffRADEC.first / interval;
  double vel_dec = diffRADEC.second / interval;
  
  return std::make_pair(vel_ra, vel_dec);
}

void convert_galLB2RAdec(std::vector<double>& world) {
  double l_val = world[0] / 180. * M_PI;
  double b_val = world[1] / 180. * M_PI;
  double cos_b = cos(b_val);
  double sin_b = sin(b_val);
  
  double cos_d_ngp = 0.8899880874849542;
  double sin_d_ngp = 0.4559837761750669;
  double l_ncp = 2.145566759798267518;
  
  world[0] = fmod((atan2(cos_b * sin(l_ncp - l_val), cos_d_ngp * sin_b - sin_d_ngp * cos_b * cos(l_ncp - l_val))
    + 3.3660332687500039) / M_PI * 180., 360.); // RA
  // need a positive modulo here
  if (world[0] < 0.0) {
    world[0] += 360.;
  }
  world[1] = asin(sin_d_ngp * sin_b + cos_d_ngp * cos_b * cos(l_ncp - l_val)) / M_PI * 180.; // dec
}

void NewAttitude::write_attitude(const std::string& filename, bool clobber) {
  
  std::unique_ptr<CCfits::FITS> outfile = sixteOpenFITSFileWrite(filename, clobber, false, "attitude file");
  
  // create an attitude extension
  std::vector<std::string> col_names = {"TIME", "RA", "DEC", "ROLLANG"};
  std::vector<std::string> col_forms = {"1D", "1D", "1D", "1D"};
  std::vector<std::string> col_units = {"s", "deg", "deg", "deg"};
  
  outfile->addTable("ATT", (int) nentries_, col_names, col_forms, col_units);
  auto &table = outfile->extension(1);
  
  // add header keywords
  table.addKey("MJDREF", mjdref_, "", false);
  table.addKey("TSTART", tstart_, "", false);
  table.addKey("TSTOP", tstop_, "", false);
  table.addKey("ALIGNMEN", "NORTH", "", false);
  table.addKey("ORIGIN", "ECAP", "", false);
  
  // write the columns
  // make empty arrays of time, ra, dec and angle
  std::vector<double> time, ra, dec, roll;
  
  // loop over attitude entries
  size_t ii;
  for (ii = 0; ii < nentries_; ii++) {
    NewAttitudeEntry entry = entries_[ii];
    // time and rollis just copied
    time[ii] = entry.time;
    roll[ii] = entry.roll_angle;
    
    // ra and dec are calculated from nz and nx;
    auto ra_dec = calculateRaDec(entry.nz);
    ra[ii] = (ra_dec.first * 180. / M_PI);
    dec[ii] = (ra_dec.second * 180. / M_PI);
  }
  
  // dump into the file
  table.column("TIME").write(time, 1);
  table.column("RA").write(ra, 1);
  table.column("DEC").write(dec, 1);
  table.column("ROLLANG").write(roll, 1);
}

// calculate a lissajous pattern from time 0 to 1, with Amplitude ampl
lissajous_pattern_struct lissajous_pattern(double amplitude, size_t nentries) {
  double fraction_a = 3.0 / 4.;
  double fraction_b = 4.0 / 4.;
  double offset = M_PI / 4.;
  
  lissajous_pattern_struct lps;
  
  for (size_t ii = 0; ii < nentries; ii++) {
    lps.time.push_back( (1.0 * (double)ii) / ((double)nentries - 1.0) ); //[0:1.0:#1000];
    lps.x_val.push_back( sin(fraction_a * lps.time[ii] * 2 * M_PI * 4 + offset) * amplitude );
    lps.y_val.push_back( sin(fraction_b * lps.time[ii] * 2 * M_PI * 4) * amplitude );
  }
  
  return lps;
}


NewAttitudeEntry read_attitude_file(long row, Fitsfile& fitsfile) {
  double time;
  double ra; //[deg]
  double dec; //[deg]
  double rollang; //[deg]
  
  fitsfile.readCol("TIME", row, time);
 
  if (fitsfile.checkColExists("RA")) fitsfile.readCol("RA", row, ra);
  else if (fitsfile.checkColExists("VIEWRA")) fitsfile.readCol("VIEWRA", row, ra);
  else throw SixteException("No column \"RA\" or \"VIEWRA\" found in attitude file");
  
  if (fitsfile.checkColExists("DEC")) fitsfile.readCol("DEC", row, dec);
  else if (fitsfile.checkColExists("VIEWDEC")) fitsfile.readCol("VIEWDEC", row, dec);
  else throw SixteException("No column \"DEC\" or \"VIEWDEC\" found in attitude file");
  
  if (fitsfile.checkColExists("ROLLANG")) fitsfile.readCol("ROLLANG", row, rollang);
  else if (fitsfile.checkColExists("ROLL")) fitsfile.readCol("ROLL", row, rollang);
  else if (fitsfile.checkColExists("ROLL1")) fitsfile.readCol("ROLL1", row, rollang);
  else {
    healog(3) << " *** warning : could not find ROLLANG/ROLL/ROLL1 field in the Attitude file,"
                 " setting roll angle to 0.0" << std::endl;
    rollang = 0.0;
  }
  
  return {time, ra, dec, rollang};
}

std::pair<double, double> NewAttitude::getRaDec (size_t row_num) {
  auto attitude_file_entry = entries_[row_num];
  auto ra_dec = calculateRaDec(attitude_file_entry.nz);
  ra_dec.first *= 180. / M_PI;
  ra_dec.second *= 180. / M_PI;
  
  return ra_dec;
}

std::pair<double, double> NewAttitude::diffRaDec (std::pair<double, double> firstRaDec,
                                                  std::pair<double, double> secondRaDec) {
  
  //difference of current and succeeding attitude entries [deg]
  double diffRA = fabs(firstRaDec.first - secondRaDec.first); //attitude-interval
  double diffDEC = fabs(firstRaDec.second - secondRaDec.second);
  
  return std::make_pair(diffRA, diffDEC);
}

NewAttitudeEntry::NewAttitudeEntry(double time_in, double ra_in, double dec_in, double rollangle_in){
  // Calculate and store attitude data:
  time = time_in;
  
  // Telescope pointing direction nz:
  nz = unitVector(ra_in * M_PI / 180., dec_in * M_PI / 180.);
  
  // Roll-Angle:
  roll_angle = rollangle_in * M_PI / 180.;
}

}
