/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/


#include "NewArf.h"
#include "SixteException.h"
#include "sixte_random.h"

#include <cstring>
#include "NewSIXT.h"

namespace sixte {

NewArf::NewArf (const std::vector<std::string>& arf_filenames) {
  int status = EXIT_SUCCESS;
  std::vector<ARF*> arfs;
  
  for (const auto & arf_filename : arf_filenames) {
    arfs.push_back(loadARF((char *)arf_filename.c_str(), &status));
    checkStatusThrow(status, "Failed to load ARF from " + arf_filename);
  }
  arf_ = getARF(&status);
  checkStatusThrow(status, "Failed to load new ARF");
  
  arf_->NumberEnergyBins = arfs[0]->NumberEnergyBins;
  arf_->HighEnergy = new float[arfs[0]->NumberEnergyBins];
  arf_->LowEnergy = new float[arfs[0]->NumberEnergyBins];
  arf_->EffArea = new float[arfs[0]->NumberEnergyBins];
  
  std::strcpy(arf_->ARFVersion, arfs[0]->ARFVersion);
  std::strcpy(arf_->Telescope, arfs[0]->Telescope);
  std::strcpy(arf_->Instrument, arfs[0]->Instrument);
  std::strcpy(arf_->Detector, arfs[0]->Detector);
  std::strcpy(arf_->Filter, arfs[0]->Filter);
  std::strcpy(arf_->ARFExtensionName, arfs[0]->ARFExtensionName);

  num_arfs_ = arfs.size();
  
  if (num_arfs_ > 1) {
    addARFsToMultiTelARF(arfs);
    getCumulARF(arfs);
    arf_energy_grid_lo_.assign(arf_->LowEnergy, arf_->LowEnergy + arf_->NumberEnergyBins);
  } else {
    for (int ii = 0; ii < arf_->NumberEnergyBins; ii++) {
      arf_->EffArea[ii] = arfs[0]->EffArea[ii];
      arf_->LowEnergy[ii] = arfs[0]->LowEnergy[ii];
      arf_->HighEnergy[ii] = arfs[0]->HighEnergy[ii];
    }
  }
  
  for (auto& arf : arfs) {
    freeARF(arf);
  }
}

NewArf::~NewArf() {
  delete[](arf_->LowEnergy);
  delete[](arf_->HighEnergy);
  delete[](arf_->EffArea);
  free(arf_);
}

void NewArf::addARFsToMultiTelARF(std::vector<ARF*> arfs) {
  for (int ii = 0; ii < arf_->NumberEnergyBins; ii++) {
    arf_->EffArea[ii] = 0.;
    arf_->LowEnergy[ii] = arfs[0]->LowEnergy[ii];
    arf_->HighEnergy[ii] = arfs[0]->HighEnergy[ii];
    for (size_t jj = 0; jj < num_arfs_; jj++) {
      if ((std::abs(arf_->LowEnergy[ii] - arfs[jj]->LowEnergy[ii]) > 1e-8) ||
          (std::abs(arf_->HighEnergy[ii] - arfs[jj]->HighEnergy[ii]) > 1e-8)) {
        //TODO: change this for all ARF sizes
        throw SixteException("ARFs must have the same binning to generate MultiTelescopeARF!");
      }
      arf_->EffArea[ii] += arfs[jj]->EffArea[ii];
    }
  }
}

void NewArf::getCumulARF(std::vector<ARF*> arfs) {
  cumulARF_ = std::vector<std::vector<double>>(arf_->NumberEnergyBins, std::vector<double>(arfs.size(), 0.0));
  
  for (int ii = 0; ii < arf_->NumberEnergyBins; ii++) {
    for (size_t telid = 0; telid < arfs.size(); telid++) {
      if (telid == 0) {
        cumulARF_[ii][telid] = arfs[telid]->EffArea[ii] / arf_->EffArea[ii];
      }
      else {
        cumulARF_[ii][telid] = cumulARF_[ii][telid - 1] + arfs[telid]->EffArea[ii] / arf_->EffArea[ii];
      }
    }
  }
}

std::vector<std::vector<double>> NewArf::retCumulARF () {
  return cumulARF_;
}

const struct ARF* NewArf::c_arf() const {
  return arf_;
}

long NewArf::findArfBinIndex(double energy) {
  auto index = findIndexBinarySearch(arf_energy_grid_lo_, static_cast<float>(energy));

  if (index < 0) {
    // Check if energy lies in last bin
    auto last_bin_index = arf_->NumberEnergyBins - 1;
    if (arf_->LowEnergy[last_bin_index] <= energy && energy <= arf_->HighEnergy[last_bin_index]) {
      return last_bin_index;
    } else {
      throw SixteException("Failed to find ARF bin for energy " + std::to_string(energy));
    }
  }

  return index;
}

size_t NewArf::getTelescopeID (double energy) {
  if (num_arfs_ == 1) return 0;

  auto bin_index = findArfBinIndex(energy);

  double rand_num = getUniformRandomNumber();
  size_t rand_telid = 0;

  while ((rand_num > cumulARF_[bin_index][rand_telid]) && ((rand_telid + 1) < num_arfs_)) {
    rand_telid++;
  }
  
  return rand_telid;
}

}