/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/
#include "Geometry.h"
#include "NewSIXT.h"
#include "SixteException.h"

namespace sixte {

void Geometry::build_transforms() {
  auto translation = SixteAffTransformation3::translation({-xrval_, -yrval_, 0.});
  auto rotation = SixteAffTransformation3::rotationZ(rota_);

  focal_to_det_ = rotation * translation;
  det_to_focal_ = focal_to_det_.inverse();
}

Geometry::Geometry(double xrval, double yrval, double rota, const std::optional<BoundingBox2d>& bounding_box_det)
: xrval_(xrval), yrval_(yrval), rota_(rota), bounding_box_det_(bounding_box_det) {
  build_transforms();
}

Geometry::Geometry(XMLData& xml_data, const std::optional<BoundingBox2d>& bounding_box_det)
: bounding_box_det_(bounding_box_det) {

  auto detector = xml_data.child("detector");
  auto parent_node = detector.optionalChild("geometry").value_or(detector);

  if (auto wcs = parent_node.optionalChild("wcs")) {
    xrval_ = wcs->attributeAsDouble("xrval");
    yrval_ = wcs->attributeAsDouble("yrval");
    rota_ = wcs->attributeAsDoubleOr("rota", 0.0) * M_PI/180.;
  } else {
    // if no WCS is found at all, use default values
    xrval_ = 0.;
    yrval_ = 0.;
    rota_ = 0.;
  }

  build_transforms();
}


SixtePoint Geometry::transformFocalToDet(const SixtePoint& pos) const {
  return focal_to_det_(pos);
}

SixtePoint Geometry::transformDetToFocal(const SixtePoint& pos) const {
  return det_to_focal_(pos);
}

bool Geometry::contains(const SixtePoint& photon_det_position) const {
  if (!bounding_box_det_) throw SixteException("Bounding box not set");
  return bounding_box_det_->containsPoint( Point_2(photon_det_position.x(), photon_det_position.y()) );
}

}
