/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "GTICollection.h"
#include "NewSIXT.h"

namespace sixte {

GTICollection::GTICollection(const std::string& gti_filename, const ObsTime& obs_time) {
  int status = EXIT_SUCCESS;
  gti_ = getGTIFromFileOrContinuous(const_cast<char *>(gti_filename.c_str()),
                                    obs_time.tstart,
                                    obs_time.tstart + obs_time.exposure,
                                    obs_time.mjdref, &status);
  checkStatusThrow(status, "Failed to initialize GTI from " + gti_filename);
}

GTICollection::GTICollection(const std::string& gti_filename) {
  int status = EXIT_SUCCESS;
  gti_ = loadGTI(const_cast<char *> (gti_filename.c_str()), &status);
  checkStatusThrow(status, "Failed to initialize GTI from " + gti_filename);
}

GTICollection::~GTICollection() {
  freeGTI(&gti_);
}

std::vector<std::pair<double, double>> GTICollection::getAllGtiBins() const {
  std::vector<std::pair<double, double>> gtibins;
  
  for (int gtibin = 0; gtibin < gti_->ngti; gtibin++) {
    gtibins.emplace_back(gti_->start[gtibin], gti_->stop[gtibin]);
  }
  return gtibins;
}

void GTICollection::filterWith(GTICollection &filter) {
  int status = EXIT_SUCCESS;

  GTI orig_copy;
  HDgti_init(&orig_copy);

  HDgti_copy(&orig_copy, gti_, &status);
  checkStatusThrow(status, "Failed copying original GTI for filtering");

  HDgti_merge(GTI_AND, gti_, &orig_copy, filter.gti_, &status);
  checkStatusThrow(status, "Failed filtering GTIs");

  HDgti_free(&orig_copy);
}

void GTICollection::saveGtiExtension(fitsfile *const fptr) const {
  int status = EXIT_SUCCESS;
  saveGTIExt(fptr, (char *) "STDGTI", gti_, &status);
  checkStatusThrow(status, "Failed to save GTI extension");
}

void GTICollection::saveGtiExtension(fitsfile *const fptr,
  std::string const& telescop, std::string const& instrume) const {

  int status = EXIT_SUCCESS;

	HDgti_write(fptr, gti_, (char *) "STDGTI",
	            (char *) "START", (char *) "STOP", &status);
  checkStatusThrow(status, "Failed to save GTI extension");

	// Move to STDGTI extension
	fits_movnam_hdu(fptr, BINARY_TBL, (char *) "STDGTI", 0, &status);

  checkStatusThrow(status, "Failed to moving to STDGTI extension");

	// Write TELESCOP and INSTRUME keywords to STDGTI extension
  std::string buf{telescop}; // need to do this copy business to get a non-const ptr
  fits_update_key(fptr, TSTRING, "INSTRUME", buf.data(), NULL, &status);
  buf = instrume; // need to do this copy business to get a non-const ptr
	fits_update_key(fptr, TSTRING, "TELESCOP", buf.data(), NULL, &status);

  checkStatusThrow(status, "Failed writing INSTRUME and TELESCOP to STDGTI extension");

}

}
