/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <algorithm>
#include <boost/dynamic_bitset.hpp>
#include <boost/random.hpp>
#include <cmath>
#include <deque>
#include <memory>
#include <utility>
#include <vector>
#include <queue>
#include "ArrayGeometry.h"
#include "Ebounds.h"
#include "NewEventfile.h"
#include "FuncBlock.h"
#include "Signal.h"
#include "sixte_random.h"
#include "XMLData.h"

namespace sixte {

const int LUT_BANDWIDTH = 14;

template<typename T>
double calcMedian(std::vector<T> vec);

template<typename T_Signal>
struct PixelSignal {
  PixelSignal(T_PixId pix, T_Signal sig)
      : pix_id{pix}, signal{sig} {}

  T_PixId pix_id;
  T_Signal signal;
};

using SliceSignal = std::vector<PixelSignal<double>>;
using LineSignal = std::vector<SliceSignal>;

void printSliceSignal(const SliceSignal& slice_signal);
void printLineSignal(const LineSignal& line_signal);

enum class DataFlags {
  BP, // Bad Pixel
  MF, // MIP or MISFIT
  ET, // Energy Threshold
  NT, // Neighbor Threshold
  PT, // Pattern Flag
  VD, // Valid Data
  DataFlags_MAX = VD // Maximum possible enum value
};

class PixelData {
 public:
  explicit PixelData(T_PixId pix_id, boost::dynamic_bitset<> signal)
      : pixel_signal_(pix_id, std::move(signal)) {
    setFlag(DataFlags::BP, true);
    setFlag(DataFlags::MF, true);
  }

  T_PixId pixID() const {
    return pixel_signal_.pix_id;
  }

  boost::dynamic_bitset<> signal() const {
    return pixel_signal_.signal;
  }

  void setSignal(boost::dynamic_bitset<> signal) {
    pixel_signal_.signal = std::move(signal);
  }

  void setSignal(unsigned long signal) {
    pixel_signal_.signal = boost::dynamic_bitset<>(pixel_signal_.signal.size(),
                                                   signal);
  }

  void addSubtractionBit() {
    pixel_signal_.signal.push_back(true);
  }

  void dropSignalMSB() {
    pixel_signal_.signal.reset(pixel_signal_.signal.size() - 1);
  }

  bool flag(DataFlags flag) const {
    return data_flags_[static_cast<int>(flag)];
  }

  void setFlag(DataFlags flag, bool value) {
    data_flags_[static_cast<int>(flag)] = value;
  }

  boost::dynamic_bitset<> flags() const {
    return data_flags_;
  }

 private:
  boost::dynamic_bitset<> data_flags_ = boost::dynamic_bitset<>(static_cast<int>(DataFlags::DataFlags_MAX) + 1, 0);
  PixelSignal<boost::dynamic_bitset<>> pixel_signal_;
};

using SliceData = std::vector<PixelData>;
using LineData = std::vector<SliceData>;

using LookUpTable = std::vector<boost::dynamic_bitset<>>;

void printSliceData(const SliceData& slice_data);
void printLineData(const LineData& line_data);

struct PixelDataPacket {
  PixelDataPacket(PixelData pixel_data_in, double time_in,
                  unsigned int xindex_in, unsigned int yindex_in,
                  long frame_number_in)
      : pixel_data(std::move(pixel_data_in)),
        time{time_in},
        xindex{xindex_in},
        yindex{yindex_in},
        frame_number{frame_number_in} {}

  PixelData pixel_data;
  double time;
  unsigned int xindex;
  unsigned int yindex;
  long frame_number;
};

class ASIC : public FuncBlock<ASIC, SliceSignal, SliceSignal> {
 public:
  explicit ASIC(double common_mode_sigma)
      : common_mode_sigma_{common_mode_sigma} {}

  SliceSignal processData_impl(SliceSignal slice_signal) {
    SliceSignal asic_out(std::move(slice_signal));

    addCommonModeOffset(asic_out);

    return asic_out;
  }

 private:
  void addCommonModeOffset(SliceSignal& slice_signal) const {
    double cm_offset = getGaussRandomNumber() * common_mode_sigma_;

    for (auto& pixel_signal: slice_signal) {
      pixel_signal.signal += cm_offset;
    }
  }

  double common_mode_sigma_;
};

class ADC : public FuncBlock<ADC, const SliceSignal&, SliceData> {
 public:
  ADC(double i_min, double i_max)
      : i_min_{i_min},
        i_max_{i_max} {
    o_min_ = 0;
    o_max_ = (unsigned long) (pow(2, ADC_BITWIDTH) - 2);
    overflow_ = o_max_ + 1;

    // Precalculate conversion factor
    conversion_factor_ = static_cast<double>(o_max_ - o_min_) / (i_max_ - i_min_);
  }

  SliceData processData_impl(const SliceSignal& slice_signal) {
    SliceData adc_out;

    for (const auto& pixel_signal: slice_signal) {
      auto adu = double2adu(pixel_signal.signal);
      adc_out.emplace_back(pixel_signal.pix_id,
                           boost::dynamic_bitset<>(ADC_BITWIDTH, adu));
    }

    return adc_out;
  }

 private:
  [[nodiscard]] unsigned long double2adu(double i_val) const {
    if ((i_val < i_min_) || (i_val > i_max_)) {
      return overflow_;
    } else {
      return o_min_ + (unsigned long) ((i_val - i_min_) * conversion_factor_);
    }
  }

  const int ADC_BITWIDTH = 14;

  double i_min_;
  double i_max_;
  unsigned long o_min_;
  unsigned long o_max_;
  unsigned long overflow_;

  double conversion_factor_;
};

enum class PipelineFlags {
  CTU,
  PCU,
  CMCU,
  EFU,
  PTAU,
  PFU,
  BypassAll,
  WriteEvents,
  PipelineFlags_MAX = WriteEvents // Maximum possible enum value
};

class CommandInterface {
 public:
  CommandInterface() {
    pipeline_flags_.set();
    setPipelineFlag(PipelineFlags::BypassAll, false);

    pfu_mask_.set();
    setPFUMask(DataFlags::VD, false);
  }

  void setOLT(const LookUpTable& offset_lookup_table) {
    offset_lookup_table_ = offset_lookup_table;
  }

  void setELT(const LookUpTable& energy_lookup_table) {
    energy_lookup_table_ = energy_lookup_table;
  }

  void setNLT(const LookUpTable& neighbour_lookup_table) {
    neighbour_lookup_table_ = neighbour_lookup_table;
  }

  boost::dynamic_bitset<> OLT(unsigned int pix_id) const {
    return offset_lookup_table_[pix_id];
  }

  boost::dynamic_bitset<> ELT(unsigned int pix_id) const {
    return energy_lookup_table_[pix_id];
  }

  boost::dynamic_bitset<> NLT(unsigned int pix_id) const {
    return neighbour_lookup_table_[pix_id];
  }

  bool pipelineFlag(PipelineFlags flag) const {
    return pipeline_flags_[static_cast<int>(flag)];
  }

  boost::dynamic_bitset<> pipelineFlags() const {
    return pipeline_flags_;
  }

  void setPipelineFlag(PipelineFlags flag, bool value) {
    pipeline_flags_[static_cast<int>(flag)] = value;
  }

  void setPFUMask(DataFlags flag, bool value) {
    pfu_mask_[static_cast<int>(flag)] = value;
  }

  void setPipelineFlags(boost::dynamic_bitset<> pipeline_flags) {
    pipeline_flags_ = std::move(pipeline_flags);
  }

  bool bypassStage(PipelineFlags flag) const {
    return !pipelineFlag(flag) || pipelineFlag(PipelineFlags::BypassAll);
  }

  boost::dynamic_bitset<> pfu_mask() const {
    return pfu_mask_;
  }

 private:
  LookUpTable offset_lookup_table_;
  LookUpTable energy_lookup_table_;
  LookUpTable neighbour_lookup_table_;

  boost::dynamic_bitset<> pipeline_flags_ =
      boost::dynamic_bitset<>(static_cast<int>(PipelineFlags::PipelineFlags_MAX) + 1);

  boost::dynamic_bitset<> pfu_mask_ =
      boost::dynamic_bitset<>(static_cast<int>(DataFlags::DataFlags_MAX) + 1);
};

// Work in Progress
class CTU : public FuncBlock<CTU, SliceData, SliceData> {
 public:
  explicit CTU(const CommandInterface& command_interface, size_t R1, size_t R2)
      : command_interface_(command_interface),
        R1_{R1},
        R2_{R2},
        subpipeline_(2, boost::dynamic_bitset<>()) {}

  SliceData processData_impl(SliceData slice_data) {
    SliceData ctu_out(std::move(slice_data));

    if (command_interface_.bypassStage(PipelineFlags::CTU)) {
      return ctu_out;
    }

    for (auto& pixel_data: ctu_out) {
      auto tmp_signal = pixel_data.signal();

      // Subtract CT correction value
      pixel_data.addSubtractionBit();
      pixel_data.setSignal(pixel_data.signal().to_ulong() - CTCorrVal());

      // Update subpipeline
      subpipeline_.pop_back();
      subpipeline_.push_front(tmp_signal);
    }

    return ctu_out;
  }

 private:
  unsigned long CTCorrVal () {
     auto p2 = R2_ * subpipeline_[1].to_ulong();
     auto p1 = R1_ * subpipeline_[0].to_ulong();

     auto P = p1 + p2;
     return P/8192;
  }

  const CommandInterface& command_interface_;
  size_t R1_;
  size_t R2_;
  std::deque<boost::dynamic_bitset<>> subpipeline_;
};

class PCU : public FuncBlock<PCU, SliceData, SliceData> {
 public:
  explicit PCU(const CommandInterface& command_interface)
      : command_interface_(command_interface) {}

  SliceData processData_impl(SliceData slice_data) {
    SliceData pcu_out(std::move(slice_data));

    if (command_interface_.bypassStage(PipelineFlags::PCU)) {
      return pcu_out;
    }

    for (auto& pixel_data: pcu_out) {
      pixel_data.addSubtractionBit();

      // Subtract offset correction value
      auto off_corr_val = command_interface_.OLT(pixel_data.pixID());
      pixel_data.setSignal(pixel_data.signal().to_ulong() - off_corr_val.to_ulong());
    }

    return pcu_out;
  }

 private:
  const CommandInterface& command_interface_;
};

class CMCU : public FuncBlock<CMCU, SliceData, SliceData> {
 public:
  explicit CMCU(const CommandInterface& command_interface)
      : command_interface_(command_interface) {}

  SliceData processData_impl(SliceData slice_data) {
    SliceData cmcu_out(std::move(slice_data));

    if (command_interface_.bypassStage(PipelineFlags::CMCU)) {
      return cmcu_out;
    }

    // Get ADU signals of the slice
    std::vector<unsigned long> slice_adu_signals;
    for (const auto& pixel_data: cmcu_out) {
      slice_adu_signals.push_back(pixel_data.signal().to_ulong());
    }

    // Calculate median
    unsigned long median = static_cast<unsigned long>(round(calcMedian(slice_adu_signals)));

    // Subtract median from all pixels of the slice
    for (auto& pixel_data: cmcu_out) {
      pixel_data.addSubtractionBit();

      pixel_data.setSignal(pixel_data.signal().to_ulong() - median);
    }

    return cmcu_out;
  }

 private:
  const CommandInterface& command_interface_;
};

class EFU : public FuncBlock<EFU, SliceData, SliceData> {
 public:
  explicit EFU(const CommandInterface& command_interface)
      : command_interface_(command_interface) {}

  SliceData processData_impl(SliceData slice_data) {
    SliceData efu_out(std::move(slice_data));

    if (command_interface_.bypassStage(PipelineFlags::EFU)) {
      return efu_out;
    }

    for (auto& pixel_data: efu_out) {
      // Check if signal is above energy and/or neighbor threshold
      if (pixel_data.signal().to_ulong() >= command_interface_.ELT(pixel_data.pixID()).to_ulong()) {
        pixel_data.setFlag(DataFlags::ET, true);
      }

      if (pixel_data.signal().to_ulong() >= command_interface_.NLT(pixel_data.pixID()).to_ulong()) {
        pixel_data.setFlag(DataFlags::NT, true);
      }

      // Drop subtraction bits
      pixel_data.dropSignalMSB();
      pixel_data.dropSignalMSB();
    }

    return efu_out;
  }

 private:
  const CommandInterface& command_interface_;
};

class PTAU : public FuncBlock<PTAU, SliceData, SliceData> {
 public:
  explicit PTAU(const CommandInterface& command_interface,
                unsigned int pix_per_slice)
      : command_interface_(command_interface),
        pix_per_slice_(pix_per_slice) {
    resetSliceBuffer();
  }

  SliceData processData_impl(SliceData slice_data) {

    if (command_interface_.bypassStage(PipelineFlags::PTAU)) {
      return slice_data;
    }

    /*
    // Fill buffer if there are not enough slices stored
    if (slice_buffer_.size() < 2) {
      slice_buffer_.push_front(std::move(slice_data));
      // Return empty slice data
      return SliceData();
    }
    */

    // Add new slice to buffer
    slice_buffer_.push_front(std::move(slice_data));

    // Run PTAU algorithm
    //std::cout << "Run PTAU algorithm\n";
    for (size_t row = 0; row < slice_buffer_[1].size(); row++) {
      //std::cout << "slice_buffer_[1][row].flag(DataFlags::ET) = " << slice_buffer_[1][row].flag(DataFlags::ET) << std::endl;
      if (slice_buffer_[1][row].flag(DataFlags::ET)) {
        bool PT_flag_value = !(row_analysis(row) || line_analysis(row));
        //std::cout << "row: " << row << ", PT_flag_value: " << PT_flag_value << std::endl;
        updatePTFlags(row, PT_flag_value);
        pixelPromotion(row);
      }
    }

    // Return line C
    SliceData line_C = slice_buffer_.back();
    slice_buffer_.pop_back();

    return line_C;
  }

  void resetSliceBuffer() {
    slice_buffer_.clear();
    for (int ii = 0; ii < 2; ii++) {
      slice_buffer_.emplace_back(pix_per_slice_, PixelData(0, boost::dynamic_bitset<>(1)));
    }
  }

 private:
  void promoteIfValidNeighborPixel(int line, int row) {
    if (slice_buffer_[line][row].flag(DataFlags::NT)) {
      slice_buffer_[line][row].setFlag(DataFlags::ET, true);
    }
  }

  void pixelPromotion(size_t row_central_pixel) {
    for (int line = 0; line < 3; line++) {
      promoteIfValidNeighborPixel(line, row_central_pixel);

      if (row_central_pixel >= 1) {
        promoteIfValidNeighborPixel(line, row_central_pixel - 1);
      }

      if (row_central_pixel < slice_buffer_.front().size() - 1) {
        promoteIfValidNeighborPixel(line, row_central_pixel + 1);
      }
    }
  }

  void updatePTFlags(size_t row_central_pixel, bool PT_flag_value) {
    for (int line = 0; line < 3; line++) {
      slice_buffer_[line][row_central_pixel].setFlag(DataFlags::PT, PT_flag_value);

      if (row_central_pixel >= 1) {
        slice_buffer_[line][row_central_pixel - 1].setFlag(DataFlags::PT, PT_flag_value);
      }

      if (row_central_pixel < slice_buffer_.front().size() - 1) {
        slice_buffer_[line][row_central_pixel + 1].setFlag(DataFlags::PT, PT_flag_value);
      }
    }
  }

  bool row_analysis(size_t row_central_pixel) {
    bool top_row = false;
    bool bottom_row = false;

    for (int line = 0; line < 3; line++) {
      if (row_central_pixel >= 1) {
        top_row = top_row || slice_buffer_[line][row_central_pixel - 1].flag(DataFlags::NT);
      }

      if (row_central_pixel < slice_buffer_.front().size() - 1) {
        bottom_row = bottom_row || slice_buffer_[line][row_central_pixel + 1].flag(DataFlags::NT);
      }
    }

    return top_row && bottom_row;
  }

  bool line_analysis(size_t row_central_pixel) {
    bool line_A = slice_buffer_[0][row_central_pixel].flag(DataFlags::NT);
    bool line_C = slice_buffer_[2][row_central_pixel].flag(DataFlags::NT);

    if (row_central_pixel >= 1) {
      line_A = line_A || slice_buffer_[0][row_central_pixel - 1].flag(DataFlags::NT);
      line_C = line_C || slice_buffer_[2][row_central_pixel - 1].flag(DataFlags::NT);
    }

    if (row_central_pixel < slice_buffer_.front().size() - 1) {
      line_A = line_A || slice_buffer_[0][row_central_pixel + 1].flag(DataFlags::NT);
      line_C = line_C || slice_buffer_[2][row_central_pixel + 1].flag(DataFlags::NT);
    }

    return line_A && line_C;
  }

  std::deque<SliceData> slice_buffer_;
  const CommandInterface& command_interface_;
  unsigned int pix_per_slice_; // Move to CommandInterface?
};

class PFU {
 public:
  explicit PFU(const CommandInterface& command_interface)
      : command_interface_(command_interface) {}

  std::vector<PixelDataPacket> processData(SliceData slice_data,
                                           unsigned int readoutindex,
                                           unsigned int xindex_first_pixel_in_slice,
                                           double readout_time,
                                           long frame_number) {
    std::vector<PixelDataPacket> pixel_data_packets;

    // Filter valid events, and add time, coordinates, and frame number.
    unsigned int xindex = xindex_first_pixel_in_slice;
    //std::cout << "PFU Mask: " << command_interface_.pfu_mask() << std::endl;
    for (auto& pixel_data: slice_data) {
      //std::cout << "PixelData Flags: " << pixel_data.flags() << std::endl;
      if (pixel_data.flags() == command_interface_.pfu_mask() || command_interface_.bypassStage(PipelineFlags::PFU)) {
        //if (pixel_data.flag(DataFlags::ET) || command_interface_.bypassStage(PipelineFlags::PFU)) {
        pixel_data.setFlag(DataFlags::VD, true);
      }

      if (pixel_data.flag(DataFlags::VD)) {
        //std::cout << "Add pixel " << xindex << " to pixel_data_packet\n";
        pixel_data_packets.emplace_back(pixel_data, readout_time,
                                        xindex, readoutindex, frame_number);
      }

      xindex++;
    }

    return pixel_data_packets;
  }

 private:
  const CommandInterface& command_interface_;
};

class EventListGenerator {
 public:
  EventListGenerator(XMLData& xml_data, const std::string& evt_filename,
                     bool clobber, unsigned int n_slices,
                     const ObsInfo& obs_info, std::shared_ptr<const Ebounds> ebounds)
      : data_fifos_(n_slices),
        ebounds_(std::move(ebounds)),
        evt_filename_{evt_filename},
        event_file_(evt_filename, clobber, xml_data, obs_info,
                    ebounds_->firstChannel(), ebounds_->numberChannels()){
    // Initialize eventfile
    

    // Set event type
    
    //ToDo: change this to CCfits
    //fits_update_key(event_file_->fptr, TSTRING, "EVTYPE", (char *) "PIXEL",
                    //"event type", &status);

    // ADU -> keV
    slope_ = (output_end_ - output_start_) / ((double) input_end_ - (double) input_start_);
  }

  void addToDataFifo(const std::vector<PixelDataPacket>& pixel_data_packets,
                     unsigned int fifo_index) {
    for (const auto& pixel_data_packet: pixel_data_packets) {
      //std::cout << "add to data FIFO!" << std::endl;
      data_fifos_[fifo_index].push_back(pixel_data_packet);
    }
  }

  void writeEvents() {
    //std::cout << "Call writeEvents" << std::endl;
    // Write all events contained in the data fifos to event file
    for (auto& data_fifo: data_fifos_) {
      //std::cout << "Data Fifo has " << data_fifo.size() << " elements" << std::endl;
      while (!data_fifo.empty()) {
        //std::cout << "write event!" << std::endl;
        auto pixel_data_package = data_fifo.front();
        data_fifo.pop_front();

        // Create temporary signal (TODO: Add correct photon metainfo; signal should be unsigned long,
        // TODO: Move adu2keV to operation strategy? Seperate column?)
        double signal_keV = adu2keV(pixel_data_package.pixel_data.signal().to_ulong());
        Signal signal(signal_keV, PhotonMetainfo(0, 0), 0); // TODO: Check creation_time

        // TODO: Use dedicated function for writing to event file.
        NewEvent event(signal, *ebounds_, pixel_data_package.xindex,
                       pixel_data_package.yindex,
                       pixel_data_package.time,
                       pixel_data_package.frame_number);
        event_file_.addEvent2CFITSIOFile(event);
      }
    }
  }

  const NewEventfile& eventFile() {
    return event_file_;
  }

  const Ebounds& ebounds() const {
    return *ebounds_;
  }

  const std::deque<PixelDataPacket>& dataFifo(int slice_number) {
    return data_fifos_[slice_number];
  }

  void clearDataFifos() {
    for (auto& data_fifo: data_fifos_) {
      data_fifo.clear();
    }
  }

 private:
  [[nodiscard]] double adu2keV(unsigned long adu_val) const {
    return output_start_ + slope_ * ((double) adu_val - input_start_);
  }

  std::vector<std::deque<PixelDataPacket>> data_fifos_;

  std::shared_ptr<const Ebounds> ebounds_;
  std::string evt_filename_;
  NewEventfile event_file_;               // Output event file

  // ADU -> keV (hardcoded)
  unsigned long input_start_ = 0;
  unsigned long input_end_ = 16383; //2^14 - 1;
  double output_start_ = 0.;
  double output_end_ = 15.;

  double slope_;
};

class EPP {
 public:
  EPP(const CommandInterface& command_interface,
      unsigned int pix_per_slice)
      : pcu_(command_interface),
        cmcu_(command_interface),
        efu_(command_interface),
        ptau_(command_interface, pix_per_slice),
        pfu_(command_interface) {}

  std::vector<PixelDataPacket> processData(SliceData& slice_data,
                                           unsigned int readoutindex,
                                           unsigned int xindex_first_pixel_in_slice,
                                           double readout_time,
                                           long frame_number) {
    //std::cout << "\n--- Run PCU ---" << std::endl;
    auto pcu_out = pcu_.processData(slice_data);
    //printSliceData(pcu_out);

    //std::cout << "\n--- Run CMCU ---" << std::endl;
    auto cmcu_out = cmcu_.processData(pcu_out);
    //printSliceData(cmcu_out);

    //std::cout << "\n--- Run EFU ---" << std::endl;
    auto efu_out = efu_.processData(cmcu_out);
    //printSliceData(efu_out);

    //std::cout << "\n--- Run PTAU ---" << std::endl;
    auto ptau_out = ptau_.processData(efu_out);
    //printSliceData(ptau_out);

    //std::cout << "\n--- Run PFU ---" << std::endl;
    auto pfu_out = pfu_.processData(ptau_out, readoutindex,
                                    xindex_first_pixel_in_slice,
                                    readout_time,
                                    frame_number);

    return pfu_out;
  }

  void resetPTAUBuffer() {
    ptau_.resetSliceBuffer();
  }

 private:
  PCU pcu_;
  CMCU cmcu_;
  EFU efu_;
  PTAU ptau_;
  PFU pfu_;
};

class FrameProcessor {
 public:
  FrameProcessor(unsigned int pix_per_slice, unsigned int n_slices,
                 XMLData& xml_data, const std::string& evt_filename,
                 bool clobber, const ObsInfo& obs_info, std::shared_ptr<const Ebounds> ebounds)
      : pix_per_slice_{pix_per_slice},
        n_slices_{n_slices},
        epps_(n_slices, EPP(command_interface_, pix_per_slice)),
        event_list_generator_(xml_data, evt_filename, clobber,
                              n_slices, obs_info, std::move(ebounds)) {}

  void processData(LineData& line_data,
                   unsigned int readoutindex,
                   double readout_time,
                   long frame_number) {
    assert(line_data.size() == epps_.size());
    //std::cout << "Run frameprocessor (frame " << frame_number << ")\n";
    //std::cout << "Pipeline Flags: " << command_interface_.pipelineFlags() << std::endl;

    if (frame_number != current_frame_number_) {
      //std::cout << "++++++++++++++++++ new frame" << std::endl;
      finishCurrentFrame();
      current_frame_number_ = frame_number;
    }

    runPipeline(line_data, readoutindex, readout_time, frame_number);
  }

  void reset() {
    clearAllBuffers();
    current_frame_number_ = 0;
  }

  void finishCurrentFrame() {
    // Finish readout of the frame
    auto original_pipeline_flags = command_interface_.pipelineFlags();
    command_interface_.setPipelineFlag(PipelineFlags::PCU, false);
    command_interface_.setPipelineFlag(PipelineFlags::CMCU, false);
    command_interface_.setPipelineFlag(PipelineFlags::EFU, false);

    LineData dummy_line(n_slices_, SliceData(pix_per_slice_, PixelData(0, boost::dynamic_bitset<>(1))));
    for (int ii = 0; ii < 2; ii++) {
      runPipeline(dummy_line, 0, 0, 0);
    }

    command_interface_.setPipelineFlags(original_pipeline_flags);

    clearAllBuffers();
  }

  void clearAllBuffers() {
    for (auto& epp: epps_) {
      epp.resetPTAUBuffer();
    }
    event_list_generator_.clearDataFifos();
  }

  const NewEventfile& eventFile() {
    return event_list_generator_.eventFile();
  }

  const Ebounds& ebounds() const {
    return event_list_generator_.ebounds();
  }

  CommandInterface& commandInterface() {
    return command_interface_;
  }

  const std::deque<PixelDataPacket>& dataFifo(int slice_number) {
    return event_list_generator_.dataFifo(slice_number);
  }

  void clearDataFifos() {
    event_list_generator_.clearDataFifos();
  }

 private:
  void runPipeline(LineData& line_data,
                   unsigned int readoutindex,
                   double readout_time,
                   long frame_number) {
    for (size_t ii = 0; ii < epps_.size(); ii++) {
      //std::cout << "Slice " << ii << std::endl;

      unsigned int xindex_first_pixel_in_slice = ii * line_data[ii].size();
      auto epp_out = epps_[ii].processData(line_data[ii], readoutindex,
                                           xindex_first_pixel_in_slice,
                                           readout_time,
                                           frame_number);

      event_list_generator_.addToDataFifo(epp_out, ii);
    }

    if (command_interface_.pipelineFlag(PipelineFlags::WriteEvents)) {
      event_list_generator_.writeEvents();
    }
  }

  unsigned int pix_per_slice_;
  unsigned int n_slices_;
  long current_frame_number_{0};

  CommandInterface command_interface_;
  std::vector<EPP> epps_;
  EventListGenerator event_list_generator_;
};

}
