/*
This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "Ebounds.h"

#include <algorithm>
#include <cmath>
#include <limits>
#include <string>

#include "SixteException.h"
#include "sixte_random.h"

namespace sixte {

Ebounds::Ebounds(std::vector<double> e_lo_keV, std::vector<double> e_hi_keV,
                 long first_channel, std::string source_filename)
    : e_lo_(std::move(e_lo_keV)),
      e_hi_(std::move(e_hi_keV)),
      first_channel_(first_channel),
      source_filename_(std::move(source_filename)) {
  if (e_lo_.size() != e_hi_.size() || e_lo_.empty()) {
    throw SixteException("Ebounds: invalid lo/hi vectors");
  }

  for (size_t ii = 0; ii < e_lo_.size(); ++ii) {
    const double lo = e_lo_[ii];
    const double hi = e_hi_[ii];

    if (!std::isfinite(lo) || !std::isfinite(hi)) {
      throw SixteException("Ebounds: non-finite boundary encountered");
    }
    if (!(lo < hi)) {
      throw SixteException("Ebounds: bin lower edge must be smaller than upper edge");
    }
    if (ii > 0) {
      if (lo < e_lo_[ii - 1]) {
        throw SixteException("Ebounds: lower edges must be non-decreasing");
      }
      if (lo < e_hi_[ii - 1]) {
        std::string msg = "Ebounds [" + source_filename_ + "]" +
               ": bins overlap at bin " + std::to_string(ii) +
               " (prev hi=" + std::to_string(e_hi_[ii - 1]) +
               ", curr lo=" + std::to_string(lo) +
               "); ensure previous high edge <= next low edge";
        throw SixteException(msg);
      }
    }
  }
}

long Ebounds::firstChannel() const { return first_channel_; }
size_t Ebounds::numberChannels() const { return e_lo_.size(); }

double Ebounds::energyLo(size_t bin) const {
  if (bin >= e_lo_.size()) {
    throw SixteException("Ebounds: energyLo bin out of range");
  }
  return e_lo_[bin];
}

double Ebounds::energyHi(size_t bin) const {
  if (bin >= e_lo_.size()) {
    throw SixteException("Ebounds: energyHi bin out of range");
  }
  return e_hi_[bin];
}

std::pair<double, double> Ebounds::sixteGetEBOUNDSEnergyLoHi(long channel) const {
  const long idx = channel - first_channel_;
  if (idx < 0 || static_cast<size_t>(idx) >= e_lo_.size()) {
    throw SixteException("Ebounds: channel out of range");
  }
  return {energyLo(static_cast<size_t>(idx)), energyHi(static_cast<size_t>(idx))};
}

double Ebounds::sixteGetEBOUNDSEnergy(long channel) const {
  auto [lo, hi] = sixteGetEBOUNDSEnergyLoHi(channel);
  auto rand = getUniformRandomNumber();
  return rand * lo + (1.0 - rand) * hi;
}

long Ebounds::channelForEnergy(double e_keV) const {
  const size_t nb = e_lo_.size();
  if (nb == 0) return -1;

  // Fast boundary checks
  if (e_keV < e_lo_.front() || e_keV > e_hi_.back()) return -1;

  // First bin whose upper edge is >= e_keV
  auto it = std::lower_bound(e_hi_.begin(), e_hi_.end(), e_keV);
  if (it == e_hi_.end()) return -1;

  const long idx = it - e_hi_.begin();
  return first_channel_ + idx;
}

long Ebounds::sixteGetEBOUNDSChannel(double e_keV) const {
  return channelForEnergy(e_keV);
}

}  // namespace sixte
