/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "Detector.h"

#include <string>
#include <string_view>

#include "NewRMF.h"

namespace sixte {

namespace {

enum class DetectorType { Microcal, CdZnTe, Ssd, Ccd, Depfet, Other };
enum class InteractionStrategy { Rmf, RmfAngle, FullAbsorption };

DetectorType parseDetectorType(std::string_view type_str) {
  if (type_str == "microcal") return DetectorType::Microcal;
  if (type_str == "CdZnTe") return DetectorType::CdZnTe;
  if (type_str == "ssd") return DetectorType::Ssd;
  if (type_str == "ccd") return DetectorType::Ccd;
  if (type_str == "depfet") return DetectorType::Depfet;

  printWarning("Unknown detector type: " + std::string(type_str) +
               "; using generic detector defaults");
  return DetectorType::Other;
}

InteractionStrategy parseInteractionStrategyAttr(const XMLNode& det) {
  // Default to rmf for backwards compatibility
  auto value = det.hasAttribute("interaction")
                   ? det.attributeAsString("interaction")
                   : "rmf";

  if (value == "rmf") return InteractionStrategy::Rmf;
  if (value == "rmf_angle") return InteractionStrategy::RmfAngle;

  throw SixteException("Unknown detector interaction: " + value);
}

// Microcal detectors always use full absorption, others read the XML attribute.
InteractionStrategy chooseInteractionStrategy(DetectorType detector_type,
                                              const XMLNode& det) {
  if (detector_type == DetectorType::Microcal) {
    return InteractionStrategy::FullAbsorption;
  }
  return parseInteractionStrategyAttr(det);
}

std::string rmfPath(const XMLData& xml_data, const XMLNode& det) {
  return xml_data.dirname() + det.child("rmf").attributeAsString("filename");
}

// Microcal detectors still need an EBOUNDS grid from an RMF, even though their
// interaction is "FullAbsorption".
std::shared_ptr<const Ebounds> loadEboundsHint(
    DetectorType detector_type, const XMLData& xml_data, const XMLNode& det) {
  if (detector_type != DetectorType::Microcal) {
    return nullptr;
  }

  const auto rmf_path = rmfPath(xml_data, det);
  return loadEboundsOnly(rmf_path);
}

std::unique_ptr<PhotonInteractionStrategy> buildInteraction(
    InteractionStrategy interaction_strategy, const XMLData& xml_data,
    const XMLNode& det, const std::shared_ptr<RmfRegistry>& rmf_registry) {
  switch (interaction_strategy) {
    case InteractionStrategy::Rmf: {
      auto rmf_path = rmfPath(xml_data, det);
      rmf_registry->load(rmf_path);
      auto rmf = rmf_registry->get(rmf_path);
      return std::make_unique<RMFBasedInteraction>(rmf);
    }
    case InteractionStrategy::RmfAngle:
      return std::make_unique<AngleDependentRMFBasedInteraction>(xml_data,
                                                                 rmf_registry);
    case InteractionStrategy::FullAbsorption:
      return std::make_unique<FullAbsorption>();
  }

  throw SixteException("Unknown detector interaction strategy");
}

// Decide the canonical EBOUNDS grid for the detector.
std::shared_ptr<const Ebounds> computeCanonicalEbounds(
    const std::shared_ptr<const Ebounds>& preset,
    const PhotonInteractionStrategy& interaction) {
  // 1) Explicit override
  if (preset) {
    return preset;
  }

  // 2) Ask the interaction strategy
  if (auto eb = interaction.canonicalEbounds()) {
    return eb;
  }

  // 3) Not available
  throw SixteException("Detector: missing canonical EBOUNDS");
}

std::unique_ptr<CarrierFormationStrategy> buildCarrier(
    DetectorType detector_type, const XMLData& xml_data) {
  switch (detector_type) {
    case DetectorType::Microcal:
      return std::make_unique<HeatPulseSimple>();
    case DetectorType::CdZnTe:
      return std::make_unique<ChargePointFactory>();
    case DetectorType::Ssd:
      return std::make_unique<ChargePointFactory>();
    case DetectorType::Ccd:
    case DetectorType::Depfet:
    case DetectorType::Other:
      return std::make_unique<GaussianSimple>(xml_data);
  }

  throw SixteException("Unknown detector type");
}

SixteException detectorException(const SixteException& e, size_t xml_id,
                                         size_t chip_id) {
  return SixteException("Detector (xml_id=" + std::to_string(xml_id) +
                        ", chip_id=" + std::to_string(chip_id) +
                        "): " + e.what());
}

}

Detector::Detector(XMLData& xml_data, const OutputFiles& outfiles,
                   const ObsInfo& obs_info, bool skip_invalids, size_t xml_id,
                   size_t chip_id, const std::shared_ptr<RmfRegistry>& rmf_registry)
    : chip_id_(chip_id) {
  try {
    const auto det = xml_data.child("detector");

    const auto detector_type = parseDetectorType(det.attributeAsString("type"));

    const auto interaction_strategy =
        chooseInteractionStrategy(detector_type, det);

    // Some detectors (microcal) override the EBOUNDS grid directly from an RMF
    auto ebounds_hint = loadEboundsHint(detector_type, xml_data, det);

    auto interaction =
        buildInteraction(interaction_strategy, xml_data, det, rmf_registry);

    canonical_ebounds_ = computeCanonicalEbounds(ebounds_hint, *interaction);

    // Configure the absorber: interaction + carrier formation
    absorber_.setPhotonInteractionStrategy(std::move(interaction));
    absorber_.setCarrierFormationStrategy(
        buildCarrier(detector_type, xml_data));

    // Construct the sensor
    sensor_ = createSensor(xml_data, obs_info, outfiles, skip_invalids, xml_id,
                           canonical_ebounds_, rmf_registry);

    // Tie absorber geometry to the sensor and set chip id
    absorber_.setGeometry(xml_data, sensor_->arrayGeometry()->boundingBox());
    absorber_.setChipId(chip_id_);

  } catch (const SixteException& e) {
    throw detectorException(e, xml_id, chip_id_);
  }
}

std::shared_ptr<ArrayGeometry> Detector::sensorGeometry() {
  return sensor_->arrayGeometry();
}

std::shared_ptr<Geometry> Detector::absorberGeometry() {
  return absorber_.geometry();
}

std::shared_ptr<const Ebounds> Detector::canonicalEbounds() const {
  return canonical_ebounds_;
}

void Detector::propagateReadout(double tstop) {
  sensor_->propagateReadout(tstop);
}

void Detector::addCarrier(const Carrier& carrier) {
  sensor_->addCarrier(carrier);
}

Carriers Detector::absorb(const SixtePhoton& photon) {
  return absorber_.absorb(photon);
}

Carriers Detector::absorbEnergy(EnergyDepositions energy_depositions) {
  return absorber_.absorbEnergy(energy_depositions);
}

void Detector::finishSimulation(double tstop) {
  sensor_->finishReadout(tstop);
}

void Detector::postprocessing(NewAttitude& attitude, const GTICollection& gti) {
  sensor_->postprocessing(absorber_, attitude, gti);
}

void Detector::clearSensor() {
  sensor_->clear();
}

void Detector::setSensorStartTime(double tstart) {
  sensor_->setStartTime(tstart);
}

}
