/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <optional>
#include <string>
#include <vector>
#include <utility>

#include "XMLData.h"

namespace sixte {
class CodedMask {
 public:
  /**
   * Create coded mask from XML.
   * @param xml_data XML containing mask parameters
   */
  explicit CodedMask(XMLData& xml_data);

  /**
   * Sample an open pixel from the mask pattern.
   * @return Mask pixel coordinates (x, y) of an open pixel
   * @throws SixteException if no open pixels are available
   */
  [[nodiscard]] std::pair<int, int> sampleOpenPixel() const;

  /**
   * Convert mask pixel coordinates to physical mask position.
   * @param x_pixel X pixel coordinate
   * @param y_pixel Y pixel coordinate
   * @param x_offset x-offset within pixel [0,1)
   * @param y_offset y-offset within pixel [0,1)
   * @return Physical position in mask plane [m]
   */
  [[nodiscard]] std::pair<double, double> pixelToPhysical(
      int x_pixel, int y_pixel, double x_offset, double y_offset) const;

  /**
   * Check whether a given physical coordinate in the mask plane is open.
   *
   * This is an O(1) lookup into the loaded mask pattern.
   *
   * @param mask_x X coordinate in mask plane [m]
   * @param mask_y Y coordinate in mask plane [m]
   * @return True if the corresponding mask pixel is open, false otherwise
   */
  [[nodiscard]] bool isOpenAtPhysical(double mask_x, double mask_y) const noexcept;

  // Getters
  [[nodiscard]] double xWidth() const noexcept { return x_width_; }
  [[nodiscard]] double yWidth() const noexcept { return y_width_; }
  [[nodiscard]] double xOffset() const noexcept { return x_offset_; }
  [[nodiscard]] double yOffset() const noexcept { return y_offset_; }
  [[nodiscard]] double xPixelSize() const noexcept { return x_pixel_size_; }
  [[nodiscard]] double yPixelSize() const noexcept { return y_pixel_size_; }
  [[nodiscard]] size_t xPixels() const noexcept { return mask_.size(); }
  [[nodiscard]] size_t yPixels() const noexcept { return mask_.empty() ? 0 : mask_[0].size(); }
  [[nodiscard]] double openFraction() const noexcept { return static_cast<double>(open_pixels_count_) / static_cast<double>(total_pixels_); }

 private:
  [[nodiscard]] std::optional<std::pair<int, int>> physicalToPixel(
      double mask_x, double mask_y) const noexcept;

  /**
   * Load coded mask pattern from FITS file.
   * @param filename Path to FITS file containing mask pattern
   * @throws SixteException if file cannot be loaded or is invalid
   */
  void loadMaskFromFits(const std::string& filename);

  /**
   * Calculate derived parameters from loaded mask data.
   * (computes pixel sizes, total/open pixel counts...)
   */
  void calculateDerivedParameters();

  /**
   * Validate mask properties and parameters.
   * @throws SixteException if mask properties are invalid
   */
  void validate() const;

  /**
   * Log mask information.
   */
  void logMaskInfo() const;

  // Mask data: mask_[x][y] where 0 = blocked, non-zero = open
  std::vector<std::vector<int>> mask_;

  // Mask parameters
  double x_width_{0.0};         ///< Physical mask width in X [m]
  double y_width_{0.0};         ///< Physical mask width in Y [m]
  double x_offset_{0.0};        ///< Mask center offset in X [m]
  double y_offset_{0.0};        ///< Mask center offset in Y [m]
  double x_pixel_size_{0.0};    ///< Physical size of one mask pixel in X [m]
  double y_pixel_size_{0.0};    ///< Physical size of one mask pixel in Y [m]
  size_t total_pixels_{0};      ///< Total number of mask pixels
  size_t open_pixels_count_{0}; ///< Number of open pixels in mask

  // Efficient storage of open pixel coordinates for direct sampling
  std::vector<std::pair<int, int>> open_pixels_;
};
}  // namespace sixte