/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "CodedMask.h"

#include <cmath>

#include "healog.h"
#include "NewSIXT.h"
#include "SixteCCFits.h"
#include "sixte_random.h"

namespace sixte {
CodedMask::CodedMask(XMLData& xml_data) {
  auto mask_node = xml_data.child("telescope").child("codedmask");

  auto size_node = mask_node.child("size");
  x_width_ = size_node.attributeAsDouble("x_width");
  y_width_ = size_node.attributeAsDouble("y_width");

  if (auto offset_node = mask_node.optionalChild("offset")) {
    x_offset_ = offset_node->attributeAsDouble("x");
    y_offset_ = offset_node->attributeAsDouble("y");
  }

  const auto mask_filename = mask_node.child("pattern").attributeAsString("filename");
  loadMaskFromFits(xml_data.dirname() + mask_filename);

  calculateDerivedParameters();
  validate();
  logMaskInfo();
}

std::pair<int, int> CodedMask::sampleOpenPixel() const {
  if (open_pixels_.empty()) {
    throw SixteException("No open pixels available in coded mask");
  }

  const auto index = static_cast<size_t>(getUniformRandomNumber() * static_cast<double>(open_pixels_.size()));
  return open_pixels_[index];
}

std::pair<double, double> CodedMask::pixelToPhysical(
    int x_pixel, int y_pixel, double x_offset, double y_offset) const {
  auto mask_x = (x_pixel + x_offset) * x_pixel_size_ - x_width_/2.0 + x_offset_;
  auto mask_y = (y_pixel + y_offset) * y_pixel_size_ - y_width_/2.0 + y_offset_;

  return {mask_x, mask_y};
}

std::optional<std::pair<int, int>> CodedMask::physicalToPixel(double mask_x, double mask_y) const noexcept {
  if (mask_.empty() || mask_[0].empty()) {
    return std::nullopt;
  }

  // Mask coordinate system:
  // pixelToPhysical() maps pixel (0,0) with offset (0,0) to
  //   (-x_width_/2 + x_offset_, -y_width_/2 + y_offset_)
  const auto x_min = -x_width_ / 2.0 + x_offset_;
  const auto y_min = -y_width_ / 2.0 + y_offset_;

  const auto x_rel = (mask_x - x_min) / x_pixel_size_;
  const auto y_rel = (mask_y - y_min) / y_pixel_size_;

  if (x_rel < 0.0 || y_rel < 0.0) {
    return std::nullopt;
  }

  const auto x_pixel = static_cast<int>(std::floor(x_rel));
  const auto y_pixel = static_cast<int>(std::floor(y_rel));

  if (x_pixel < 0 || y_pixel < 0) {
    return std::nullopt;
  }

  if (x_pixel >= static_cast<int>(xPixels()) || y_pixel >= static_cast<int>(yPixels())) {
    return std::nullopt;
  }

  return std::make_pair(x_pixel, y_pixel);
}

bool CodedMask::isOpenAtPhysical(double mask_x, double mask_y) const noexcept {
  const auto pixel = physicalToPixel(mask_x, mask_y);
  if (!pixel) {
    return false;
  }
  return mask_[pixel->first][pixel->second] != 0;
}

void CodedMask::loadMaskFromFits(const std::string& filename) {
  healog(5) << "Loading coded mask from " << filename << std::endl;

  Fitsfile fitsfile{filename, FileMode::read};
  fitsfile.moveToPHDU();

  int naxis1{}, naxis2{};
  fitsfile.readKey("NAXIS1", naxis1);
  fitsfile.readKey("NAXIS2", naxis2);

  if (naxis1 <= 0 || naxis2 <= 0) {
    throw SixteException("Invalid mask dimensions: " + std::to_string(naxis1) +
                        " x " + std::to_string(naxis2));
  }

  std::vector<int> mask_data;
  const std::vector<long> fpixel{1, 1};

  fitsfile.readPix(fpixel, naxis1 * naxis2, mask_data);

  // Convert to 2D vector mask_[x][y]
  mask_.resize(naxis1);
  for (auto xx = 0; xx < naxis1; ++xx) {
    mask_[xx].resize(naxis2);
    for (auto yy = 0; yy < naxis2; ++yy) {
      mask_[xx][yy] = mask_data[xx + yy * naxis1];
    }
  }
}

void CodedMask::calculateDerivedParameters() {
  x_pixel_size_ = x_width_ / static_cast<double>(xPixels());
  y_pixel_size_ = y_width_ / static_cast<double>(yPixels());

  total_pixels_ = xPixels() * yPixels();
  open_pixels_count_ = 0;

  for (auto xx = 0; xx < static_cast<int>(xPixels()); ++xx) {
    for (auto yy = 0; yy < static_cast<int>(yPixels()); ++yy) {
      if (mask_[xx][yy] != 0) {
        ++open_pixels_count_;
        open_pixels_.emplace_back(xx, yy);
      }
    }
  }
}

void CodedMask::validate() const {
  if (mask_.empty() || mask_[0].empty()) {
    throw SixteException("Coded mask empty or invalid");
  }

  if (x_width_ <= 1e-10 || y_width_ <= 1e-10 || x_width_ > 10.0 || y_width_ > 10.0) {
    throw SixteException("Invalid mask dimensions: " + std::to_string(x_width_) +
                        " x " + std::to_string(y_width_) + " m");
  }

  if (open_pixels_count_ == 0) {
    throw SixteException("Coded mask has no open pixels");
  }

  if (open_pixels_count_ == total_pixels_) {
    printWarning("Coded mask has all pixels open");
  }

  if (openFraction() < 0.2 || openFraction() > 0.8) {
    printWarning("Unusual mask open fraction: " + std::to_string(openFraction()));
  }
}

void CodedMask::logMaskInfo() const {
  healog(5) << "Coded mask loaded successfully:" << std::endl;
  healog(5) << "  Mask: " << xPixels() << "x" << yPixels()
            << " pixels, " << x_width_ << "x" << y_width_ << " m" << std::endl;
  healog(5) << "  Pixel size: " << x_pixel_size_ << "x" << y_pixel_size_ << " m" << std::endl;
  healog(5) << "  Open fraction: " << openFraction() << std::endl;
  healog(5) << "  Open pixels: " << open_pixels_count_ << " / " << total_pixels_ << std::endl;

  if (x_offset_ != 0.0 || y_offset_ != 0.0) {
    healog(0) << "  Mask offset: (" << x_offset_ << ", "
              << y_offset_ << ") m" << std::endl;
  }
}
}  // namespace sixte