/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <cmath>
#include <utility>
#include <algorithm>
#include <vector>
#include <stdexcept>
#include <map>
#include <memory>
#include <variant>
#include <deque>
#include <queue>
#include <gsl/gsl_sf_erf.h>
#include "BoundingBox.h"
#include "Cache.h"
#include "EnergyDeposition.h"
#include "Geometry.h"
#include "PhotonMetainfo.h"
#include "Polygon.h"
#include "SixteException.h"
#include "XMLData.h"

namespace sixte {

// as a general note:
// Coordinates in a Carrier object are always in detector units,
// not in focal plane units
// Specifically, this means that the coordinates are relative to the XRPIX and YRPIX,
// with rotation already applied
class Carrier {
 public:
  virtual ~Carrier() = default;
  [[nodiscard]] virtual double charge(const BoundingBox2d& bbox) const = 0;
  [[nodiscard]] virtual const BoundingBox2d& bounding_box() const = 0;

  [[nodiscard]] virtual PhotonMetainfo photon_metainfo() const = 0;
  [[nodiscard]] virtual double creationTime() const = 0;
};

using CarrierPtr = std::shared_ptr<Carrier>;

struct CarrierPtrCmp {
  bool operator() (const CarrierPtr& carrier_a, const CarrierPtr& carrier_b) const {
    return carrier_a->creationTime() > carrier_b->creationTime();
  }
};

using Carriers = std::priority_queue<CarrierPtr, std::vector<CarrierPtr>, CarrierPtrCmp>;

CarrierPtr pop_next_carrier(Carriers& carriers);

void transferCarriers(Carriers& source, Carriers& destination);

class GaussianCCSimple: public Carrier {
 public:
  GaussianCCSimple(const EnergyDeposition& energy_deposition, double sigma,
                   double creation_time);

  /**
   * Calculate charge within a given polygon.
   *
   * @param poly   A closed polygon.
   * @return       The charge within the given polygon.
   */
  [[nodiscard]] double charge(const BoundingBox2d& bbox) const override;

  [[nodiscard]] const BoundingBox2d& bounding_box() const override;;

  [[nodiscard]] PhotonMetainfo photon_metainfo() const override;

  [[nodiscard]] double creationTime() const override;

 private:
  /**
   * Get distance from the charge cloud center to closest edge of a rectangular
   * pixel in x direction.
   *
   * @param rectangular_pixel    A rectangular pixel.
   * @return                     Distance from charge cloud center to closest
   *     edge of rectangular_pixel in x direction. The return value is negative
   *     if the cloud center is inside the pixel, positive otherwise.
   */
  [[nodiscard]] double minDistX(const BoundingBox2d &rectangular_pixel) const;

  /**
   * Get distance from the charge cloud center to closest edge of a rectangular
   * pixel in y direction.
   *
   * @param rectangular_pixel    A rectangular pixel.
   * @return                     Distance from charge cloud center to closest
   *     edge of rectangular_pixel in y direction. The return value is negative
   *     if the cloud center is inside the pixel, positive otherwise.
   */
  [[nodiscard]] double minDistY(const BoundingBox2d &rectangular_pixel) const;

  static constexpr int cc_extent = 3; // Three sigma (used to calculate bbox)

  double sigma_;        ///< Standard deviation of Gaussian charge cloude [m].
  SixtePoint center_;      ///< The position of the charge cloud center.
  double radius_;       ///< Charge cloud radius (3*sigma) [m].

  BoundingBox2d bounding_box_; ///< The bounding bounding_box of the charge cloud.
  double total_charge_; ///< Total charge within charge cloud.

  /// Meta information about source photon (ph_id, src_id, tel_id)
  PhotonMetainfo photon_metainfo_;

  double creation_time_;
};


class ChargePoint: public Carrier {
 public:
  ChargePoint() = default;

  explicit ChargePoint(const EnergyDeposition& energy_deposition);

  /**
   * Calculate charge within a given polygon.
   *
   * @param poly   A closed polygon.
   * @return       The charge within the given polygon.
   */
  [[nodiscard]] double charge(const BoundingBox2d& bbox) const override;

  [[nodiscard]] const BoundingBox2d& bounding_box() const override;;

  [[nodiscard]] PhotonMetainfo photon_metainfo() const override;

  [[nodiscard]] double creationTime() const override;

 private:
  SixtePoint center_;      ///< The position of the absorption.

  BoundingBox2d bounding_box_; ///< The bounding bounding_box of the carrier.
  ///< This is a box of size zero!
  double total_charge_{}; ///< Total energy, in units of keV

  /// Meta information about source photon (ph_id, src_id)
  PhotonMetainfo photon_metainfo_{};

  double creation_time_{};
};


class ExponentialCC: public Carrier {
};

class GaussianCCPhys: public Carrier {
};

class HeatPulse: public Carrier {
 public:
  HeatPulse(const EnergyDeposition& energy_deposition,
                   double creation_time);

  /**
   * Calculate charge within a given polygon.
   *
   * @param poly   A closed polygon.
   * @return       The charge within the given polygon.
   */
  [[nodiscard]] double charge(const BoundingBox2d& bbox) const override;

  [[nodiscard]] const BoundingBox2d& bounding_box() const override;;

  [[nodiscard]] PhotonMetainfo photon_metainfo() const override;

  [[nodiscard]] double creationTime() const override {
    return creation_time_;
  }

 private:
  SixtePoint center_;      ///< The position of the absorption.

  BoundingBox2d bounding_box_; ///< The bounding bounding_box of the pulse.
                               ///< This is a box of size zero!
  double heat_; ///< Total energy, in units of keV

  /// Meta information about source photon (ph_id, src_id)
  PhotonMetainfo photon_metainfo_;

  double creation_time_;

};

class ElectronTrack: public Carrier {
};


class CarrierFormationStrategy {
 public:
  virtual ~CarrierFormationStrategy() = default;
  virtual Carriers createCarriers(const Geometry& absorber_geometry,
                                  EnergyDepositions&& energy_depositions) = 0;
};

class GaussianSimple: public CarrierFormationStrategy {
 public:
  explicit GaussianSimple(const XMLData& xml_data);

  Carriers createCarriers(const Geometry& absorber_geometry,
                          EnergyDepositions&& energy_depositions) override;

 private:
  // Charge cloud sigma as function of the photon energy.
  [[nodiscard]] double ccSigma(double energy) const;

  double cloud_par1_{0.}; // First parameter of charge cloud size.
  double cloud_par2_{0.}; // Second parameter of charge cloud size.
};

class ChargePointFactory: public CarrierFormationStrategy {
 public:
  Carriers createCarriers(const Geometry& absorber_geometry,
                          EnergyDepositions&& energy_depositions) override;
};

class HeatPulseSimple: public CarrierFormationStrategy {
 public:
  Carriers createCarriers(const Geometry& absorber_geometry,
                          EnergyDepositions&& energy_depositions) override;
};

}
