/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "Carrier.h"

namespace sixte {

GaussianSimple::GaussianSimple(const XMLData& xml_data) {
  // Initialize charge cloud parameters. At least one of them must be given.
  auto split = xml_data.child("detector").child("split");
  bool par1_given = split.hasAttribute("par1");
  bool par2_given = split.hasAttribute("par2");
  
  if (!par1_given && !par2_given) {
    throw SixteException("Failed to initialize charge cloud parameters"
                         " (par1/par2) from XML.");
  }
  
  if (par1_given) {
    cloud_par1_ = split.attributeAsDouble("par1");
  }
  if (par2_given) {
    cloud_par2_ = split.attributeAsDouble("par2");
  }
}

Carriers GaussianSimple::createCarriers(const Geometry& absorber_geometry, EnergyDepositions&& energy_depositions) {
  Carriers carriers;
  for (auto& energy_deposition: energy_depositions) {
    energy_deposition.position_ = absorber_geometry.transformFocalToDet(energy_deposition.position_);
    carriers.push(std::make_shared<GaussianCCSimple>(energy_deposition,
                                                     ccSigma(energy_deposition.energy_),
                                                     energy_deposition.creation_time_));
  }

  return carriers;
}

double GaussianSimple::ccSigma(double energy) const {
  return cloud_par1_ + cloud_par2_ * sqrt(energy);
}

double gaussInt(double x) {
  // Static cache. Saves return values of last two calls to gaussInt.
  static Cache cache;

  // Check if x is in the cache.
  double cached_val{0};
  if (cache.contains(x, cached_val))
    return cached_val;

  // Check if -x is in the cache.
  if (cache.contains(-x, cached_val))
    return 1 - cached_val;

  // Not in cache. Calculate integral and save new (argument,value) pair in cache.
  double gauss_int = gsl_sf_erf_Q(x);
  cache.add(std::make_pair(x, gauss_int));
  return gauss_int;
}

double GaussianCCSimple::minDistX(const BoundingBox2d &rectangular_pixel) const {
  // Distance to left pixel edge
  double dist_left = rectangular_pixel.bottom_left().x() -
      center_.x();

  // Distance to right pixel edge
  double dist_right = center_.x() -
      rectangular_pixel.top_right().x();

  return std::abs(dist_left) < std::abs(dist_right) ? dist_left : dist_right;
}

double GaussianCCSimple::minDistY(const BoundingBox2d &rectangular_pixel) const {
  // Distance to lower pixel edge
  double dist_low = rectangular_pixel.bottom_left().y() -
      center_.y();

  // Distance to upper pixel edge
  double dist_up = center_.y() -
      rectangular_pixel.top_right().y();

  return std::abs(dist_low) < std::abs(dist_up) ? dist_low : dist_up;
}

double GaussianCCSimple::charge(const BoundingBox2d& bbox) const {
  // If the charge cloud does not extend into the pixel, return 0.
  if (!bounding_box().doOverlap(bbox))
    return 0;

  // Initialize integrals to 1.
  double gauss_integral_x{1};
  double gauss_integral_y{1};

  // Calculate distance from charge cloud center to closest pixel edge in
  // x direction.
  double min_dist_x = minDistX(bbox);
  // If this distance is smaller than the charge cloud radius, the charge cloud
  // extends over the pixel edge. Calculate corresponding integral.
  if (std::abs(min_dist_x) < radius_)
    gauss_integral_x = gaussInt(min_dist_x / sigma_);

  // Same in y direction
  double min_dist_y = minDistY(bbox);
  if (std::abs(min_dist_y) < radius_)
    gauss_integral_y = gaussInt(min_dist_y / sigma_);

  return gauss_integral_x * gauss_integral_y * total_charge_;
}

GaussianCCSimple::GaussianCCSimple(const EnergyDeposition& energy_deposition, double sigma,
                                   double creation_time)
    : sigma_{sigma},
      center_{energy_deposition.position_},
      radius_{cc_extent * sigma_},
      bounding_box_{Point_2{center_.x(), center_.y()}, 2 * radius_},
      total_charge_{energy_deposition.energy_},
      photon_metainfo_{energy_deposition.photon_metainfo_},
      creation_time_{creation_time}{}

const BoundingBox2d& GaussianCCSimple::bounding_box() const {
  return bounding_box_;
}

PhotonMetainfo GaussianCCSimple::photon_metainfo() const {
  return photon_metainfo_;
}

double GaussianCCSimple::creationTime() const {
  return creation_time_;
}

// TODO: Factory Template?
Carriers ChargePointFactory::createCarriers(const Geometry& absorber_geometry,
                                            EnergyDepositions&& energy_depositions) {
  Carriers carriers;

  for (auto& energy_deposition: energy_depositions) {
    energy_deposition.position_ = absorber_geometry.transformFocalToDet(energy_deposition.position_);
    carriers.push(std::make_shared<ChargePoint>(energy_deposition));
  }

  return carriers;
}

Carriers HeatPulseSimple::createCarriers(const Geometry& absorber_geometry,
                                         EnergyDepositions&& energy_depositions) {
  Carriers carriers;

  for (auto& energy_deposition: energy_depositions) {
    energy_deposition.position_ = absorber_geometry.transformFocalToDet(energy_deposition.position_);
    carriers.push(std::make_unique<HeatPulse>(energy_deposition,
                                              energy_deposition.creation_time_));
  }

  return carriers;
}

HeatPulse::HeatPulse(const EnergyDeposition& energy_deposition,
                     double creation_time)
  : center_{energy_deposition.position_},
    bounding_box_{Point_2{center_.x(), center_.y()}, 0.},
    heat_{energy_deposition.energy_},
    photon_metainfo_{energy_deposition.photon_metainfo_},
    creation_time_{creation_time} {}

double HeatPulse::charge(const BoundingBox2d &bbox) const {
  // since the heat pulse's bounding box is just a point,
  // no need to overlap - just use the center
  if (!bbox.containsPoint(bounding_box().center())) {
    return 0;
  } else {
    return heat_;
  }
}

const BoundingBox2d& HeatPulse::bounding_box() const {
  return bounding_box_;
}

PhotonMetainfo HeatPulse::photon_metainfo() const {
  return photon_metainfo_;
}

ChargePoint::ChargePoint(const EnergyDeposition& energy_deposition)
    : center_{energy_deposition.position_},
      bounding_box_{Point_2{center_.x(), center_.y()}, 0.},
      total_charge_{energy_deposition.energy_},
      photon_metainfo_{energy_deposition.photon_metainfo_},
      creation_time_{energy_deposition.creation_time_} {}

double ChargePoint::charge(const BoundingBox2d &bbox) const {
  if (!bounding_box().doOverlap(bbox)) {
    return 0;
  } else {
    return total_charge_;
  }
}

const BoundingBox2d& ChargePoint::bounding_box() const {
  return bounding_box_;
}

PhotonMetainfo ChargePoint::photon_metainfo() const {
  return photon_metainfo_;
}

double ChargePoint::creationTime() const {
  return creation_time_;
}

CarrierPtr pop_next_carrier(Carriers& carriers) {
  if (carriers.empty()) {
    throw SixteException("Carrier queue is empty");
  }
  auto next_carrier = carriers.top();
  carriers.pop();
  return next_carrier;
}

void transferCarriers(Carriers& source, Carriers& destination) {
  while (!source.empty()) {
    destination.push(source.top());
    source.pop();
  }
}


}
