/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <utility>

#include "ArrayGeometry.h"
#include "Carrier.h"
#include "SixtePhoton.h"
#include "XMLData.h"
extern "C" {
#include "impact.h"
#include "mxs.h"
}

namespace sixte {

/** \class CalibrationSource
 *
 * Abstract base class for X-ray calibration sources. Derived classes implement
 * the concrete type of photon generation (realised via CRTP).
 */
template <typename Derived>
class CalibrationSource {
 protected:
  ~CalibrationSource() = default;

 public:
  /**
   * Generates the next photon from the calibration source within a given time
   * interval.
   *
   * @param dt The considered time interval.
   * @return[opt] The next photon from the calibration source, or std::nullopt
   *              if the photon lies outside the given time interval.
   */
  std::optional<SixtePhoton> getNextPhoton(std::pair<double, double> dt);
};


/** \class ModulatedXraySource
 *
 * Simulates a modulated X-ray source (mxs), characterized by
 * - frequency [Hz] of the mxs flashes [Hz],
 * - duration [s] of the mxs flashes,
 * - count rate [cps] on the detector during the mxs flashes (modeled as a
 *   Poisson process).
 *
 * The MXS photon energies are sampled from an event list (using inverse
 * transform sampling), and impact positions are sampled from an illumination map.
 */
class ModulatedXraySource: public CalibrationSource<ModulatedXraySource> {
 public:
  /**
    * Create a new MXS object with parameters from the XML file.
    * @brief Constructor.
    *
    * @param xml_data The XML file that contains the MXS parameters.
    * @param absorber_array_geometry A shared pointer to the absorber array
    *                                geometry.
    */
  explicit ModulatedXraySource(XMLData& xml_data,
                               std::shared_ptr<ArrayGeometry> absorber_array_geometry);

  /**
   * @brief MXS implementation of getNextPhoton.
   */
  std::optional<SixtePhoton> getNextPhoton(std::pair<double, double> dt);

 private:
  /**
   * Updates the current flash interval if necessary so that the given start
   * value lies in it (necessary when going to a new GTI).
   *
   * @param tstart The start time of the currently considered time interval.
   */
  void updateFlashInterval(double tstart);

  /**
   * Generates an mxs photon impact with a given impact time.
   *
   * @param impact_time The impact time of the generated mxs photon.
   * @return The mxs photon impact.
   */
  SixtePhoton genMXSImpact(double impact_time);

  /**
   * Samples a random impact position on the absorber array, based on the
   * illumination map.
   *
   * @return An impact position on the absorber array.
   */
  SixtePoint sampleMXSPosition();

  /**
  * @brief Custom deleter for MXSparams struct.
  */
  struct MXSparamsDeleter {
    void operator()(MXSparams* mxs_params) const {
      freeMXSParams(&mxs_params);
    }
  };

  std::unique_ptr<MXSparams, MXSparamsDeleter> mxs_params_; ///< MXS parameters
  std::shared_ptr<ArrayGeometry> absorber_array_geometry_; ///< Absorber array geometry

  double flash_start_time_{0.}; ///< Start time of current mxs flash.
  double flash_end_time_{0.}; ///< End time of current mxs flash.
};

}

