/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "CalibrationSource.h"

namespace sixte {

template<typename Derived>
std::optional<SixtePhoton> CalibrationSource<Derived>::getNextPhoton(std::pair<double, double> dt) {
  return static_cast<Derived&>(*this).getNextPhoton(dt);
}

ModulatedXraySource::ModulatedXraySource(XMLData& xml_data, std::shared_ptr<ArrayGeometry> absorber_array_geometry)
    : absorber_array_geometry_(std::move(absorber_array_geometry)) {
  auto mxs = xml_data.child("detector").child("mxs");
  auto mxs_filename = xml_data.dirname() + mxs.attributeAsString("filename");
  auto mxs_flash_frequency = mxs.attributeAsDouble("flash_frequency");
  auto mxs_flash_duration = mxs.attributeAsDouble("flash_duration");
  auto mxs_rate = mxs.attributeAsDouble("rate");

  int status = EXIT_SUCCESS;
  mxs_params_.reset(loadMXSparams(mxs_flash_frequency, mxs_flash_duration,
                                  mxs_rate, const_cast<char*>(mxs_filename.c_str()),
                                  &status));
  checkStatusThrow(status, "Failed to load MXS parameters");
}

std::optional<SixtePhoton> ModulatedXraySource::getNextPhoton(std::pair<double, double> dt) {
  updateFlashInterval(dt.first);

  int status = EXIT_SUCCESS;
  auto time_of_next_mxs_impact = getNextMXSImpactTime(mxs_params_.get(), &flash_start_time_,
                                                      &flash_end_time_, &status);
  checkStatusThrow(status, "Failed to get MXS photon impact time");

  if (time_of_next_mxs_impact <= dt.second) {
    return genMXSImpact(time_of_next_mxs_impact);
  } else {
    return std::nullopt;
  }
}

void ModulatedXraySource::updateFlashInterval(double tstart) {
  while (flash_start_time_ < tstart) {
    flash_start_time_ += 1./ mxs_params_->mxs_frequency;
  }
  flash_end_time_ = flash_start_time_ + mxs_params_->mxs_flash_duration;
}

SixtePhoton ModulatedXraySource::genMXSImpact(double impact_time) {
  int status = EXIT_SUCCESS;
  auto energy = getMXSEnergy(mxs_params_->mxs_eventlist->energy,
                             mxs_params_->mxs_eventlist->n_events,
                             &status);
  checkStatusThrow(status, "Failed to get MXS photon energy");
  auto position = sampleMXSPosition();

  return SixtePhoton(impact_time, energy, position,
                     PhotonMetainfo(PhId::mxs_id, SrcType::mxs));
}

SixtePoint ModulatedXraySource::sampleMXSPosition() {
  int status = EXIT_SUCCESS;
  double mxs_pos_x, mxs_pos_y;

  do {
    // Get a random position from the image [pixel].
    double xd;
    double yd;
    drawRndPosFromImg(mxs_params_->mxs_img, &xd, &yd, &status);
    checkStatusThrow(status, "Failed to get MXS photon impact position");

    // Get corresponding impact position [m]
    mxs_pos_x = (xd - mxs_params_->mxs_img->wcs->crpix[0]) * mxs_params_->mxs_img->wcs->cdelt[0];
    mxs_pos_y = (yd - mxs_params_->mxs_img->wcs->crpix[1]) * mxs_params_->mxs_img->wcs->cdelt[1];

  } while ( !absorber_array_geometry_->getPixId(Point_2(mxs_pos_x, mxs_pos_y)) );

  return SixtePoint(mxs_pos_x, mxs_pos_y, 0);
}

}