/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once


#include <deque>
#include <stddef.h>

namespace sixte {

double reldiff(double a, double b);

class Cache {
 public:
  /**
   * Default constructor
   */
  Cache() = default;

  /**
   * Create a cache with given size.
   *
   * @param max_size    The size of the cache, i.e., the maximum number of
   *                    simultaneously cached elements.
   */
  Cache(size_t max_size);

  /**
   * Check whether the cache contains a given element.
   *
   * @param[in]  x    The element to look for.
   * @param[out] val  The corresponding value.
   * @return          True if the cache contains the element; false otherwise.
   */
  bool contains(double x, double& val) const;

  /**
   * Add a new element to the cache. If the cache is full, the oldest element
   * will be removed.
   *
   * @param pair    An (element, value) pair.
   */
  void add(const std::pair<double, double>& pair);

 private:
  std::deque<std::pair<double, double>> cached_vals_; ///< The cached values
  size_t max_size_{2}; ///< Size of the cache

  /// Maximum relative difference for which two values are classified as
  /// identical.
  double eps_{1e-9};
};

}
