/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "Cache.h"
#include <cmath>
#include <algorithm>

namespace sixte {

/**
 * Calculate the relative difference between two values.
 *
 * @param a   First value.
 * @param b   Second value.
 * @return    The relative difference between a and b.
 */
double reldiff(double a, double b) {
  double divisor = std::max(std::abs(a), std::abs(b));
  // If divisor is zero, both x and y are zero, so the difference between
  // them is zero.
  return divisor == 0.0 ? 0.0 : std::abs(a - b) / divisor;
}

Cache::Cache(size_t max_size) : max_size_{max_size} {}

bool Cache::contains(double x, double& val) const {
  auto it = std::find_if(std::begin(cached_vals_), std::end(cached_vals_),
                    [&](const std::pair<double, double>& pair) {
                      return reldiff(pair.first, x) < eps_;
                    });
  if (it != std::end(cached_vals_)) {
    val = (*it).second;
    return true;
  } else {
    return false;
  }
}

void Cache::add(const std::pair<double, double>& pair) {
  // If maximum size of the cache is reached, delete the first (i.e., oldest)
  // cached value.
  if (cached_vals_.size() == max_size_)
    cached_vals_.pop_front();

  // Add new value pair to cache
  cached_vals_.push_back(pair);
}

}