/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <algorithm>
#include <cmath>
#include <optional>
#include <stdexcept>
#include <utility>
#include "Polygon.h"

#include "NewSIXT.h"

namespace sixte {

class BoundingBox {
 public:
  /**
   * Create a rectangular bounding bounding_box with given corners.
   *
   * @param bottom_left   The bottom left corner of the bounding bounding_box.
   * @param top_right     The top right corner of the bounding bounding_box (x and y
   *     coordinates of top right corner both must be greater than x and y
   *     coordinates of bottom left corner).
   */
  BoundingBox(const Point_2& bottom_left, const Point_2& top_right)
  : rect_{bottom_left, top_right}{
  }

  /**
   * Create a quadratic bounding bounding_box with given center position and edge length.
   *
   * @param center_pos    The position of the bounding bounding_box center.
   * @param edge_length   The edge length of the bounding bounding_box.
   */
  BoundingBox(const Point_2& center_pos, double edge_length)
      : rect_{center_pos - Vector_2(0.5 * edge_length, 0.5 * edge_length),
              center_pos + Vector_2(0.5 * edge_length, 0.5 * edge_length)} {
  }

  /**
   * Default constructor.
   */
  BoundingBox() = default;

  /**
   * Get the bottom left corner of the bounding bounding_box.
   *
   * @return    The bottom left corner of the bounding bounding_box.
   */
  const Point_2 bottom_left() const {
    return rect_.vertex(0);
  }

  /**
   * Get the bottom right corner of the bounding bounding_box.
   *
   * @return    The bottom right corner of the bounding bounding_box.
   */
  const Point_2 bottom_right() const {
    return rect_.vertex(1);
  }

  /**
   * Get the top right corner of the bounding bounding_box.
   *
   * @return    The top right corner of the bounding bounding_box.
   */
  const Point_2 top_right() const {
    return rect_.vertex(2);
  }

  /**
   * Get the top left corner of the bounding bounding_box.
   *
   * @return    The top left corner of the bounding bounding_box.
   */
  const Point_2 top_left() const {
    return rect_.vertex(3);
  }

  /**
   * Get the center point of the bounding bounding_box.
   *
   * @return    The center point of the bounding bounding_box.
   */
  Point_2 center() const {
    return CGAL::midpoint(bottom_left(), top_right());
  }

  /**
   * Get the height of the bounding bounding_box.
   *
   * @return    The height of the bounding bounding_box.
   */
  double height() const {
    return std::abs(rect_.ymax()-rect_.ymin());
  }

  /**
   * Get the width of the bounding bounding_box.
   *
   * @return    The width of the bounding bounding_box.
   */
  double width() const {
    return std::abs(rect_.xmax()-rect_.xmin());
  }

  /**
   * Get the minimum x coordinate of the bounding box.
   */
  double xmin() const {
    return rect_.xmin();
  }

  /**
   * Get the maximum x coordinate of the bounding box.
   */
  double xmax() const {
    return rect_.xmax();
  }

  /**
   * Get the minimum y coordinate of the bounding box.
   */
  double ymin() const {
    return rect_.ymin();
  }

  /**
   * Get the maximum y coordinate of the bounding box.
   */
  double ymax() const {
    return rect_.ymax();
  }

  /**
   * Get the area of the bounding box. Returns 0 for degenerate boxes.
   */
  double area() const {
    const double dx = xmax() - xmin();
    const double dy = ymax() - ymin();
    if (dx <= 0.0 || dy <= 0.0) {
      return 0.0;
    }
    return dx * dy;
  }

  /**
   * Check whether the bounding box overlaps with another bounding bounding_box.
   *
   * @param test_box    The bounding_box to perform the check with.
   * @return            True if the boxes overlap; false otherwise (just
   *                    touching returns false).
   */
  bool doOverlap(const BoundingBox& test_box) const {
    return CGAL::do_overlap(test_box.get_bbox(), get_bbox());
  }

  /**
   * Check whether the bounding box overlaps with another bounding box
   * with strict positive area (touching edges does not count).
   *
   * This avoids CGAL's do_overlap behavior that treats touching as overlap.
   */
  bool doOverlapStrict(const BoundingBox& test_box) const {
    return (xmax() > test_box.xmin()) &&
           (test_box.xmax() > xmin()) &&
           (ymax() > test_box.ymin()) &&
           (test_box.ymax() > ymin());
  }

  /**
   * Compute the intersection of two bounding boxes.
   * Returns std::nullopt if there is no positive-area overlap.
   */
  std::optional<BoundingBox> intersection(const BoundingBox& other) const {
    const double min_x = std::max(xmin(), other.xmin());
    const double max_x = std::min(xmax(), other.xmax());
    const double min_y = std::max(ymin(), other.ymin());
    const double max_y = std::min(ymax(), other.ymax());

    if (max_x <= min_x || max_y <= min_y) {
      return std::nullopt;
    }

    return BoundingBox(Point_2(min_x, min_y), Point_2(max_x, max_y));
  }

  /**
   * Check whether the bounding box contains a point.
   *
   * @param test_box    The bounding_box to perform the check with.
   * @return            True if the boxes contains the point; false
   *                    otherwise (being on the edge returns true).
   */
  bool containsPoint(const Point_2& test_point) const {
    return !rect_.has_on_unbounded_side(test_point);
  }


 private:

  const CGAL::Bbox_2& get_bbox() const{
   if (!bbox_.has_value()) {
    bbox_ = rect_.bbox();
   }
  return bbox_.value();
  };

  cgal_Kernel::Iso_rectangle_2 rect_;
  mutable std::optional<CGAL::Bbox_2> bbox_; // LAZILY EVALUATED! Only access via get_bbox()!
};

BoundingBox translate(const BoundingBox& box, const Vector_2& vec);

// Typedef for easier usage
typedef BoundingBox BoundingBox2d;
typedef BoundingBox Rectangle2d;

}
