/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "BoundingBox.h"
#include "Geometry.h"
#include "NewAttitude.h"
#include "Polygon.h"
#include "XMLData.h"
#include "SixteCCFits.h"
#include <vector>

namespace sixte {

using T_PixId = size_t;

std::pair<double,double> detToSky(double detx, double dety, double focal_length,
  double time, NewAttitude& ac, const Geometry& absorber_geometry);

/**
 * Remove Overlapping rectangles from a list of rectangles
 * In case of overlaps, the pixel located further toward the front
 * of the array is removed (the "newest" pixels are kept)
 *
 * @param position    Vector of Rectangle2d
 * @return            Vector of rectangles without overlap
 */

std::vector<Rectangle2d> removeOverlaps(const std::vector<Rectangle2d> &pixels);

class ArrayGeometry {
 public:
  /**
   * Virtual destructor
   */
  virtual ~ArrayGeometry() = default;

  /**
   * Gets ID of pixel located at given position.
   *
   * @param position    A position on the array.
   * @return            The ID of the pixel at given position.
   */
  [[nodiscard]] virtual std::optional<T_PixId> getPixId(const Point_2& position) const = 0;

  // TODO: Make this virtual in RectangularArray
  //virtual Polygon_2 getPolygon(T_PixId pixid) = 0;

  /**
   * Gets IDs of pixels that overlap with a given bounding bounding_box.
   *
   * @param box    A 2D bounding bounding_box.
   * @return       The IDs of all pixels that overlap with the bounding bounding_box.
   */
  [[nodiscard]] virtual std::vector<T_PixId> getPixIds(const BoundingBox2d& box) const = 0;

  [[nodiscard]] virtual T_PixId getRandomPixId() const = 0;
  [[nodiscard]] virtual double totalSurfaceArea() const = 0;

  /**
   * Get a random position in detector coordinates within the given pixel
   *
   * @param pix_id   Pixel ID
   * @ return        A pair of values corresponding to [detx, dety]
   */
  [[nodiscard]] virtual std::pair<double,double> getRandPosInPixel(T_PixId pix_id) const = 0;

  /**
   * Gets the polygon surrounding a given pixel
   *
   * @param pix_id    The ID of a pixel.
   * @return          The surrounding polygon.
   */
  [[nodiscard]] virtual Rectangle2d getPolygon(T_PixId pix_id) const = 0;


  /**
   * Gets number of pixels in the array.
   *
   * @return    The number of pixels in the array.
   */
  [[nodiscard]] virtual size_t numpix() const = 0;

  /**
   * Perform photon projection on event file
   * Iterates over entries of the event file and writes into RA and DEC columns
   *
   * @param ac             Simulation attitude
   * @param focal_length   Telescope focal length
   * @param fptr           CFITSIO file pointer to event file
   * @param tstart         time after which to do photon projection
   * @param tstop          time until which to do photon projection
   */
  virtual void doPhotonProjection(NewAttitude& ac, const Geometry& absorber_geometry,
    double focal_length, fitsfile *fptr, double tstart, double tstop) const = 0;

  virtual const BoundingBox2d& boundingBox() const = 0;
};

class RectangularArray : public ArrayGeometry {
 public:
  /**
    * Creates a rectangular array.
    *
    * @param data    Parameters of the rectangular array.
    */
  explicit RectangularArray(XMLData& xml_data);

  /**
   * Destructor
   */
  ~RectangularArray() override = default;

  /**
   * Gets ID of pixel located at given position. IDs start at 0.
   *
   * @param position    A position on the array.
   * @return            The ID of the pixel at given position.

   */
  [[nodiscard]] std::optional<T_PixId> getPixId(const Point_2& position) const override;

  /**
   * Gets xi and yi of pixel located at given position.
   *
   * @param position    A position on the array.
   * @return            The xi and yi of the pixel at given position as pair.
   */
    std::optional<std::pair<size_t, size_t>> getXY(const Point_2& position) const;

  /**
   * Gets ID of pixel at coordinate (x,y)
   *
   * @param xi    The X coordinate of the pixel.
   * @param xy    The Y coordinate of the pixel.
   */

  [[nodiscard]] T_PixId xy2PixId(unsigned int xi, unsigned int yi) const;

  /**
   * Gets a random pixel ID.
   */
  [[nodiscard]] T_PixId getRandomPixId() const override;

  [[nodiscard]] virtual std::pair<double,double> getRandPosInPixel(T_PixId pix_id) const override;

  /**
   * Gets the total surface area of all pixels.
   */
  [[nodiscard]] double totalSurfaceArea() const override;

  /**
   * Gets x,y coordinates of a given pixel.
   *
   * @param pix_id    The ID of a pixel.
   * @return          The x,y coordinates of the pixel.
   */


  [[nodiscard]] std::pair<unsigned int, unsigned int> PixId2xy(T_PixId pix_id) const;

  /**
   * Gets the polygon surrounding a given pixel
   *
   * @param pix_id    The ID of a pixel.
   * @return          The surrounding polygon.
   */
  [[nodiscard]] Rectangle2d getPolygon(T_PixId pix_id) const override;

  /**
   * Gets IDs of pixels that overlap with a given bounding bounding_box.
   *
   * @param box    A 2D bounding bounding_box.
   * @return       The IDs of all pixels that overlap with the bounding bounding_box.
   */
  [[nodiscard]] std::vector<T_PixId> getPixIds(const BoundingBox2d& box) const override;

  /**
   * Gets number of pixels in the array.
   *
   * @return    The number of pixels in the array.
   */
  [[nodiscard]] size_t numpix() const override;

  [[nodiscard]] int getXWidth() const;

  [[nodiscard]] int getYWidth() const;

  void doPhotonProjection(NewAttitude& ac, const Geometry& absorber_geometry,
    double focal_length, fitsfile *fptr, double tstart, double tstop) const override;

  [[nodiscard]] const BoundingBox2d& boundingBox() const override;

 private:
  // Prototype pixel at (x, y) = (0, 0)
  Rectangle2d rectangle_;

  // Detector dimensions. Width and height [pixels].
  int xwidth_, ywidth_;

  // Reference pixel.
  double xrpix_, yrpix_;

  // Pixel width [m], including (twice) the pixel border.
  double xdelt_, ydelt_;

  // Pixel border [m]. In this area along the corner the pixel is
  // insensitive to incident photons.
  double xborder_{0.};
  double yborder_{0.};

  double total_surface_area_;

  Rectangle2d bounding_box_;
};


class FreeGeometry : public ArrayGeometry {
  public:
  /**
    * Creates an array with freely placed, rectangular pixels
    *
    * @param data    Parameters of the rectangular array.
    */
  explicit FreeGeometry(XMLData& xml_data);

  /**
   * Destructor
   */
  ~FreeGeometry() override = default;

  /**
   * Gets ID of pixel located at given position.
   *
   * @param position    A position on the array.
   * @return            The ID of the pixel at given position.
   */
  [[nodiscard]] std::optional<T_PixId> getPixId(const Point_2& position) const override;

  /**
   * Gets IDs of pixels that overlap with a given bounding bounding_box.
   *
   * @param box    A 2D bounding bounding_box.
   * @return       The IDs of all pixels that overlap with the bounding bounding_box.
   */
  [[nodiscard]] std::vector<T_PixId> getPixIds(const BoundingBox2d& box) const override;

  /**
   * Gets a random pixel ID.
   */
  [[nodiscard]] T_PixId getRandomPixId() const override;

  [[nodiscard]] virtual std::pair<double,double> getRandPosInPixel(T_PixId pix_id) const override;

  /**
   * Gets the total surface area of all pixels.
   */
  [[nodiscard]] double totalSurfaceArea() const override;

  /**
   * Gets the polygon surrounding a given pixel
   *
   * @param pix_id    The ID of a pixel.
   * @return          The surrounding polygon.
   */
  [[nodiscard]] Rectangle2d getPolygon(T_PixId pix_id) const override;

  /**
   * Gets number of pixels in the array.
   *
   * @return    The number of pixels in the array.
   */
  [[nodiscard]] size_t numpix() const override;

  void doPhotonProjection(NewAttitude& ac, const Geometry& absorber_geometry,
    double focal_length, fitsfile *fptr, double tstart, double tstop) const override;

  [[nodiscard]] const BoundingBox2d& boundingBox() const override;

 private:
  // Pixel shapes
  std::vector<Rectangle2d> pixels_;

  double total_surface_area_{0.};

  Rectangle2d bounding_box;
};

/* TODO:
class HexagonalArray: public ArrayGeometry {
public:
  // Destructor
  ~HexagonalArray() = default;

private:
  std::vector<Vector_2> translations;
};
*/

std::unique_ptr<ArrayGeometry> createGeometry(XMLData& xml_data);

}
