/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2015 Philippe Peille, IRAP
*/

#ifndef PIXEVENTFILE_H
#define PIXEVENTFILE_H 1

#include "sixt.h"
#include "event.h"

////////////////////////////////////////////////////////////////////////
// Type declarations.
////////////////////////////////////////////////////////////////////////

typedef struct {
	/** Current size of the list */
	int size;

	/** Current size of the energy/grade lists */
	int size_energy;

	/** Current end index of the list */
	int index;

	/** Index arrival time of the photons inside a record */

	//int * event_indexes;
	double * event_indexes;

	/** Pulse height of the photons */
	double * pulse_heights;

	/** Pulse grade */
	int * grading;

	/** Energy of the photons */
	double * energies;

	/** Grade 1: distance to next pulse */
	int * grades1;

	/** Grade 2: distance to previous pulse */
	int * grades2;

	/** PH_ID of the reconstructed photons */
	long * ph_ids;

} PixEventList;

typedef struct {
  /* Event time*/
  double time;

  /* Pulse height / channel id */
  int pha;

  /* Event energy */
  double energy;

  /* Pixel ID*/
  int pixid;

  /* Grades (for impacts before and after) */
  int grade1, grade2;

  /* Sky projection and location on detector*/
  double ra, dec;
  double detx, dety;

  /* Event grade */
  int grading;

  /* Photon and source IDS */
  long ph_id;
  long src_id;

  /** Split pattern type: single (1), double (2), triple (3),
      border+single (-1), border+double (-2), border+triple (-3)
      invalid pattern (0) */
  int type;

  /** Pile-up flag. */
  int pileup;

} PixEvent;


typedef struct {
	/** Pointer to the FITS file. */
	fitsfile* fptr;

	/** Number of the current row in the FITS file. The numbering
	starts at 1 for the first line. If row is equal to 0, no row
	has been read or written so far. */
	long row;

	/** Total number of rows */
	long nrows;

	/** Column numbers for time, energy, grade1, grade2, pixID, RA and DEC columns */
	int timeCol,phaCol,energyCol,grade1Col,grade2Col,pixIDCol,phIDCol,raCol,decCol,detxCol,detyCol,gradingCol,srcIDCol,typeCol, pileupCol;

} PixEventFile;

////////////////////////////////////////////////////////////////////////
// Function declarations.
////////////////////////////////////////////////////////////////////////

/** PixEventFile constructor. Returns a pointer to an empty PixEventFile data
    structure. */
PixEventFile* newPixEventFile(int* const status);

/** Create and open a new PixEventFile. */
PixEventFile* openNewPixEventFile(const char* const filename,
				  SixtStdKeywords* keywords,
				  const char clobber,
				  int* const status);

/** Opens a TES event file with the given mode */
PixEventFile* openPixEventFile(const char* const filename,const int mode, int* const status);

/** PixEventFile Destructor. */
void freePixEventFile(PixEventFile** file, int* const status);

/** Add empty event */
void addEmptyPixEvent(PixEventFile* file, int* const status);

/** Updates the event in specified row */
void updatePixEvent(PixEventFile* file, long row, double time, int pha, double signal,
                    int pixid ,int grade1,int grade2,int grading,int ph_id,
                    int src_id, int type, int pileup, int* const status);

/** Add event */
void addPixEvent(PixEventFile* file, double time, int pha, double signal,
                 int pixid, int grade1, int grade2, int grading, int ph_id,
                 int src_id, int type, int pileup, int* const status);

/** Updates the RA, DEC and DETX/Y columns with the given coordinates */
void updatePixEvtRaDecDetXY(PixEventFile* file, int row,
                            double ra, double dec,
                            double detx, double dety,
                            int* const status);

/** Get a PixEvent from a file **/
void getPixEventFromFile(PixEventFile* file, int row,
                         PixEvent* evt,
                         int* const status);

#endif /* PIXEVENTFILE_H */
