/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2015 Philippe Peille, IRAP
*/

#include "pixeventfile.h"
#include "event.h"


/** PixEventFile constructor. Returns a pointer to an empty PixEventFile data
    structure. */
PixEventFile* newPixEventFile(int* const status){
	PixEventFile* file=malloc(sizeof(*file));
	if (NULL==file) {
		*status=EXIT_FAILURE;
		SIXT_ERROR("memory allocation for PixEventFile failed");
		return(file);
	}

	// Initialize pointers with NULL.
	file->fptr    =NULL;

	// Initialize values.
	file->row  	    =1;
	file->nrows     =0;
	file->timeCol   =1;
	file->phaCol    =2;
	file->energyCol =3;
	file->pixIDCol  =4;
	file->grade1Col =5;
	file->grade2Col =6;
	file->raCol     =7;
	file->decCol    =8;
	file->detxCol   =9;
	file->detyCol   =10;
	file->gradingCol=11;
	file->phIDCol   =12;
	file->srcIDCol  =13;
  file->typeCol   =14;
  file->pileupCol   =15;

	return(file);
}

/** PixEventFile Destructor. */
void freePixEventFile(PixEventFile** file, int* const status){
	if (NULL!=*file) {
		if (NULL!=(*file)->fptr) {
			fits_close_file((*file)->fptr, status);
			CHECK_STATUS_VOID(*status);
			headas_chat(5, "closed PixEventFile list file\n");
		}
		free(*file);
		*file=NULL;
	}
}

/** Create and open a new PixEventFile. */
PixEventFile* openNewPixEventFile(const char* const filename,
				  SixtStdKeywords* keywords,
				  const char clobber,
				  int* const status){
	PixEventFile* file = newPixEventFile(status);
	CHECK_STATUS_RET(*status, file);

	int exists;
	char buffer[MAXFILENAME];
	sprintf(buffer,"%s",filename);
	fits_file_exists(buffer, &exists, status);
	CHECK_STATUS_RET(*status,file);
	if (0!=exists) {
		if (0!=clobber) {
			// Delete the file.
			remove(buffer);
		} else {
			// Throw an error.
			char msg[MAXMSG];
			sprintf(msg, "file '%s' already exists", buffer);
			SIXT_ERROR(msg);
			*status=EXIT_FAILURE;
			CHECK_STATUS_RET(*status,file);
		}
	}
	fits_create_file(&(file->fptr),buffer, status);
	CHECK_STATUS_RET(*status,file);
	int logic=(int)'T';
	int bitpix=8;
	int naxis=0;
	fits_update_key(file->fptr, TLOGICAL, "SIMPLE", &(logic), NULL, status);
	fits_update_key(file->fptr, TINT, "BITPIX", &(bitpix), NULL, status);
	fits_update_key(file->fptr, TINT, "NAXIS", &(naxis), NULL, status);
	sixt_add_fits_stdkeywords(file->fptr,1,keywords,status);
	CHECK_STATUS_RET(*status,file);

	//Write XML into header
	//char comment[MAXMSG];
	//sprintf(comment, "XMLFILE: %s", xmlfile);
	//fits_write_comment(file->fptr, comment, status);
	//CHECK_STATUS_RET(*status,file);

	// Create table

	//first column TIME
	char   *ttype[]={"TIME","PHA","SIGNAL","PIXID",
                         "GRADE1","GRADE2",
                         "RA","DEC","DETX","DETY",
                         "GRADING","PH_ID","SRC_ID","TYPE", "PILEUP"};
	char *tform[]={"1D","1J","1D","1J",
                       "1J","1J",
                       "1D","1D","1D","1D",
                       "1I","1J","1I","1I","1I"};
	char *tunit[]={"s","channel","keV", "",
                       "","",
                       "deg","deg","m","m",
                       "","","","",""};

	fits_create_tbl(file->fptr, BINARY_TBL, 0, 15,
			ttype, tform, tunit,"EVENTS", status);
	sixt_add_fits_stdkeywords(file->fptr,2,keywords,status);
	CHECK_STATUS_RET(*status,file);

	int firstpix=0,lastpix=0,numberpix=0;
	float monoen=-1.;
	long nes_tot=0,net_tot=0;
	fits_update_key(file->fptr, TINT, "FIRSTPIX", &firstpix, "First pixel in record file", status);
	fits_update_key(file->fptr, TINT, "LASTPIX", &lastpix, "Last pixel in record file", status);
	fits_update_key(file->fptr, TINT, "NPIX", &numberpix, "Number of pixels in record file", status);
	fits_update_key(file->fptr, TFLOAT, "MONOEN", &monoen, "Monochromatic energy of photons [keV]", status);
	fits_update_key(file->fptr, TLONG, "NESTOT", &nes_tot, "Total number of events simulated", status);
	fits_update_key(file->fptr, TLONG, "NETTOT", &net_tot, "Total number of events actually triggered", status);
	CHECK_STATUS_RET(*status,file);

	return(file);
}

/** Opens a PIX event file with the given mode */
PixEventFile* openPixEventFile(const char* const filename,const int mode, int* const status){
	PixEventFile * file = newPixEventFile(status);
	headas_chat(3, "open TES event file '%s' ...\n", filename);
	fits_open_table(&file->fptr, filename, mode, status);
	CHECK_STATUS_RET(*status, NULL);

	// Determine the row numbers.
	fits_get_num_rows(file->fptr, &(file->nrows), status);

	// Determine the column numbers.
	fits_get_colnum(file->fptr, CASEINSEN, "TIME", &file->timeCol, status);
	fits_get_colnum(file->fptr, CASEINSEN, "PHA", &file->phaCol, status);
	fits_get_colnum(file->fptr, CASEINSEN, "SIGNAL", &file->energyCol, status);
	fits_get_colnum(file->fptr, CASEINSEN, "GRADE1", &file->grade1Col, status);
	if (*status==COL_NOT_FOUND) {
	  file->grade1Col=-1;
	  *status=0;
	}

	fits_get_colnum(file->fptr, CASEINSEN, "GRADE2", &file->grade2Col, status);
	if (*status==COL_NOT_FOUND) {
	  file->grade2Col=-1;
	  *status=0;
	}

	fits_get_colnum(file->fptr, CASEINSEN, "PIXID", &file->pixIDCol, status);
	if (*status==COL_NOT_FOUND) {
	  file->pixIDCol=-1;
	  *status=0;
	}
	fits_get_colnum(file->fptr, CASEINSEN, "PH_ID", &file->phIDCol, status);
	if (*status==COL_NOT_FOUND) {
	  file->phIDCol=-1;
	  *status=0;
	}
	fits_get_colnum(file->fptr, CASEINSEN, "RA", &file->raCol, status);
	fits_get_colnum(file->fptr, CASEINSEN, "DEC", &file->decCol, status);

	fits_get_colnum(file->fptr, CASEINSEN, "DETX", &file->detxCol, status);
	if (*status==COL_NOT_FOUND) {
	  file->detxCol=-1;
	  *status=0;
	}
	fits_get_colnum(file->fptr, CASEINSEN, "DETY", &file->detyCol, status);
	if (*status==COL_NOT_FOUND) {
	  file->detyCol=-1;
	  *status=0;
	}

	fits_get_colnum(file->fptr, CASEINSEN, "GRADING", &file->gradingCol, status);
	if (*status==COL_NOT_FOUND) {
	  file->gradingCol=-1;
	  *status=0;
	}
	fits_get_colnum(file->fptr, CASEINSEN, "SRC_ID", &file->srcIDCol, status);
	if (*status==COL_NOT_FOUND) {
	  file->srcIDCol=-1;
	  *status=0;
	}
  fits_get_colnum(file->fptr, CASEINSEN, "TYPE", &file->typeCol, status);
	if (*status==COL_NOT_FOUND) {
	  file->typeCol=-1;
	  *status=0;
	}
  fits_get_colnum(file->fptr, CASEINSEN, "PILEUP", &file->pileupCol, status);
	if (*status==COL_NOT_FOUND) {
	  file->pileupCol=-1;
	  *status=0;
	}

	CHECK_STATUS_RET(*status, NULL);

	return(file);
}

/** Add empty event */
void addEmptyPixEvent(PixEventFile* file, int* const status) {
  addPixEvent(file, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, status);
}

/** Updates the event in specified row */
void updatePixEvent(PixEventFile* file, long row, double time, int pha, double signal,
                    int pixid ,int grade1,int grade2,int grading,int ph_id,
                    int src_id, int type, int pileup, int* const status) {
  // Error checking
  CHECK_STATUS_VOID(*status);

  //Update time column
	fits_write_col(file->fptr, TDOUBLE, file->timeCol,
			row, 1, 1, &time, status);

	//Update PHA column
	fits_write_col(file->fptr, TLONG, file->phaCol,
			row, 1, 1, &pha, status);

	//Update energy column
	fits_write_col(file->fptr, TDOUBLE, file->energyCol,
			row, 1, 1, &signal, status);

	//Update grade1 column
	fits_write_col(file->fptr, TLONG, file->grade1Col,
			row, 1, 1, &grade1, status);

	//Update grade2 column
	fits_write_col(file->fptr, TLONG, file->grade2Col,
			row, 1, 1, &grade2, status);

	//Update grading column
	fits_write_col(file->fptr, TINT, file->gradingCol,
			row, 1, 1, &grading, status);

	//Update PIXID column
	fits_write_col(file->fptr, TLONG, file->pixIDCol,
			row, 1, 1, &pixid, status);

	//Update PH_ID column
	fits_write_col(file->fptr, TLONG, file->phIDCol,
			row, 1, 1, &ph_id, status);

	//Update SRC_ID column
	fits_write_col(file->fptr, TINT, file->srcIDCol,
			row, 1, 1, &src_id, status);

  //Update TYPE column
	fits_write_col(file->fptr, TINT, file->typeCol,
			row, 1, 1, &type, status);

  //Update PILEUP column
	fits_write_col(file->fptr, TINT, file->pileupCol,
		  row, 1, 1, &pileup, status);
}

/** Add event */
void addPixEvent(PixEventFile* file, double time, int pha, double signal,
                 int pixid ,int grade1,int grade2,int grading,int ph_id,
                 int src_id, int type, int pileup, int* const status) {

	//Save time column
	fits_write_col(file->fptr, TDOUBLE, file->timeCol,
			file->row, 1, 1, &time, status);
	CHECK_STATUS_VOID(*status);

	//Save PHA column
	fits_write_col(file->fptr, TLONG, file->phaCol,
			file->row, 1, 1, &pha, status);
	CHECK_STATUS_VOID(*status);

	//Save energy column
	double energy = signal;
	fits_write_col(file->fptr, TDOUBLE, file->energyCol,
			file->row, 1, 1, &signal, status);
	CHECK_STATUS_VOID(*status);

	//Save grade1 column
	if (file->grade1Col!=-1) {
	  fits_write_col(file->fptr, TLONG, file->grade1Col,
			 file->row, 1, 1, &grade1, status);
	  CHECK_STATUS_VOID(*status);
	}

	//Save grade2 column
	if (file->grade2Col!=-1) {
	  fits_write_col(file->fptr, TLONG, file->grade2Col,
			 file->row, 1, 1, &grade2, status);
	  CHECK_STATUS_VOID(*status);
	}

	//Save grading column
	if (file->gradingCol!=-1) {
	  fits_write_col(file->fptr, TINT, file->gradingCol,
			 file->row, 1, 1, &grading, status);
	  CHECK_STATUS_VOID(*status);
	}

	//Save PIXID column
	fits_write_col(file->fptr, TLONG, file->pixIDCol,
			file->row, 1, 1, &pixid, status);
	CHECK_STATUS_VOID(*status);

	//Save PH_ID column
	fits_write_col(file->fptr, TLONG, file->phIDCol,
			file->row, 1, 1, &ph_id, status);
	CHECK_STATUS_VOID(*status);

	//Save SRC_ID column
	if (file->srcIDCol!=-1) {
	  fits_write_col(file->fptr, TINT, file->srcIDCol,
			 file->row, 1, 1, &src_id, status);
	  CHECK_STATUS_VOID(*status);
	}

  //Save TYPE column
	if (file->typeCol!=-1) {
	  fits_write_col(file->fptr, TINT, file->typeCol,
			 file->row, 1, 1, &type, status);
	  CHECK_STATUS_VOID(*status);
	}

  //Save PILEUP column
  if (file->pileupCol!=-1) {
	  fits_write_col(file->fptr, TINT, file->pileupCol,
			 file->row, 1, 1, &pileup, status);
	  CHECK_STATUS_VOID(*status);
	}

	file->row++;
	file->nrows++;
}

/** Updates the RA, DEC and DETX/Y columns with the given coordinates */
void updatePixEvtRaDecDetXY(PixEventFile* file, int row,
                            double ra, double dec,
                            double detx, double dety,
                            int* const status){
	double dbuffer=ra*180./M_PI;
	fits_write_col(file->fptr, TDOUBLE, file->raCol,
                       row, 1, 1,&dbuffer, status);
	CHECK_STATUS_VOID(*status);
	dbuffer=dec*180./M_PI;
	fits_write_col(file->fptr, TDOUBLE, file->decCol,
                       row, 1, 1,&dbuffer, status);
	CHECK_STATUS_VOID(*status);
	fits_write_col(file->fptr, TDOUBLE, file->detxCol,
                       row, 1, 1,&detx, status);
	CHECK_STATUS_VOID(*status);
	fits_write_col(file->fptr, TDOUBLE, file->detyCol,
                       row, 1, 1,&dety, status);
	CHECK_STATUS_VOID(*status);
}

/** Get a PixEvent from a file **/
void getPixEventFromFile(PixEventFile* file, int row,
                         PixEvent* evt,
                         int* const status) {

        // Check if the file has been opened.
        CHECK_NULL_VOID(file, *status, "pixevent file not open");
        CHECK_NULL_VOID(file->fptr, *status, "pixevent file not open");

        // Check if there is still a row available.
        if (row>file->nrows) {
          *status=EXIT_FAILURE;
          SIXT_ERROR("pixevent file contains no further entries");
          return;
        }

        // read the data
        // Read in the data.
        int anynul=0;
        double dnull=0.;
        long lnull=0;
        int inull=0;

	fits_read_col(file->fptr, TDOUBLE, file->timeCol, row, 1, 1,
                      &dnull, &evt->time, &anynul, status);
	//Save PHA
	fits_read_col(file->fptr, TLONG, file->phaCol,row, 1, 1,
                      &lnull, &evt->pha, &anynul, status);

	//Save energy
	fits_read_col(file->fptr, TDOUBLE, file->energyCol,row, 1, 1,
                      &dnull, &evt->energy, &anynul, status);

	//Save grade1
	if (file->grade1Col!=-1) {
          fits_read_col(file->fptr, TLONG, file->grade1Col,row, 1, 1,
                        &lnull, &evt->grade1, &anynul, status);

	}

	//Save grade2
	if (file->grade2Col!=-1) {
          fits_read_col(file->fptr, TLONG, file->grade2Col,row, 1, 1,
                        &lnull, &evt->grade2, &anynul, status);

	}

	//Save grading
	if (file->gradingCol!=-1) {
          fits_read_col(file->fptr, TINT, file->gradingCol,row, 1, 1,
                        &inull, &evt->grading, &anynul, status);

	}

	//Save PIXID
        fits_read_col(file->fptr, TLONG, file->pixIDCol,row, 1, 1,
                      &lnull, &evt->pixid, &anynul, status);


	//Save PH_ID
        fits_read_col(file->fptr, TLONG, file->phIDCol,row, 1, 1,
                      &lnull, &evt->ph_id, &anynul, status);


	//Save SRC_ID
	if (file->srcIDCol!=-1) {
          fits_read_col(file->fptr, TINT, file->srcIDCol,row, 1, 1,
                        &inull, &evt->src_id, &anynul, status);

	}

  //Save TYPE
	if (file->typeCol!=-1) {
          fits_read_col(file->fptr, TINT, file->typeCol,row, 1, 1,
                        &inull, &evt->type, &anynul, status);

	}

  //Save PILEUP
	if (file->pileupCol!=-1) {
          fits_read_col(file->fptr, TINT, file->pileupCol,row, 1, 1,
                        &inull, &evt->pileup, &anynul, status);

	}

}


void copyPixEventFile(const PixEventFile* const src,
		                  PixEventFile* const dest,
		                  const float threshold_lo_keV,
		                  const float threshold_up_keV,
		                  int* const status)
{
  // Check if the event file is empty.
  if (dest->nrows > 0) {
    *status=EXIT_FAILURE;
    SIXT_ERROR("destination event file is not empty");
    return;
  }

  // Copy the event type.
  char evtype[MAXMSG], comment[MAXMSG];
  fits_read_key(src->fptr, TSTRING, "EVTYPE", evtype, comment, status);
  if (EXIT_SUCCESS!=*status) {
    SIXT_ERROR("could not read FITS keyword 'EVTYPE'");
    return;
  }
  fits_update_key(dest->fptr, TSTRING, "EVTYPE", evtype, comment, status);
  CHECK_STATUS_VOID(*status);

  // Get memory for buffers.
  PixEvent* event=(PixEvent*)malloc(sizeof(PixEvent));
  CHECK_STATUS_VOID(*status);

  // Loop over all rows in the event file.
  for (long row = 0; row < src->nrows; row++) {

    // Read an event from the input list.
    getPixEventFromFile(src, row+1, event, status);
    CHECK_STATUS_BREAK(*status);

    // Apply the lower event threshold.
    if (event->energy < threshold_lo_keV) {
      continue;
    }

    // Apply the upper event threshold.
    if ((threshold_up_keV > 0.0) && (event->energy > threshold_up_keV)) {
      continue;
    }

    // Add the new event to the output file.
    addPixEvent(dest, event->time, event->pha, event->energy, event->pixid,
                event->grade1, event->grade2, event->grading, event->ph_id,
                event->src_id, event->type, event->pileup, status);
    CHECK_STATUS_BREAK(*status);
  }
  CHECK_STATUS_VOID(*status);

  // Free memory.
  free(event);
}
